package org.ow2.opensuit.cel.impl.tree.impl.ast;

import java.lang.reflect.Type;

import org.ow2.opensuit.cel.ICompilationContext;
import org.ow2.opensuit.cel.ICompilationMessage;
import org.ow2.opensuit.cel.IEvaluationContext;
import org.ow2.opensuit.cel.ITypeConverter;
import org.ow2.opensuit.cel.impl.ICompilationResultWriter;
import org.ow2.opensuit.cel.impl.misc.BooleanOperations;
import org.ow2.opensuit.cel.impl.misc.NumberOperations;
import org.ow2.opensuit.cel.impl.misc.RegExpBuilder;
import org.ow2.opensuit.cel.impl.tree.ExpressionEvaluationException;


public class AstUnaryOperation extends AstRightValue
{
	private ITypeConverter converter;
	private final Operator operator;

	private final AstNode child;

	public AstUnaryOperation(int position, AstNode child, AstUnaryOperation.Operator operator)
	{
		super(position);
		this.child = child;
		this.operator = operator;
	}

	public boolean compile(ITypeConverter converter, ICompilationContext ctx, ICompilationResultWriter messages)
	{
		this.converter = converter;
		
		// --- 1: compile child
		boolean compiled = child.compile(converter, ctx, messages);
		
		if (!compiled){
			return false;
		}

		// --- 2: compile this (check type)
		if (!operator.checkType(converter, child.getType()))
		{
			messages.addMessage(this, ICompilationMessage.ERROR_LEVEL, "Unsupported type for operation "+operator+".");
			return false;
		}
		return true;
	}

	public Class<?> getType()
	{
		return operator.getType(child.getType());
	}

	public Type getGenericType()
	{
		return operator.getType(child.getType());
	}

	public boolean isStaticValue()
	{
		return child.isStaticValue();
	}

	public Operator getOperator()
	{
		return operator;
	}

	public Object invoke(IEvaluationContext context) throws Exception
	{
		try
		{
			return operator.apply(converter, child.invoke(context));
		}
		catch (Exception e)
		{
			throw new ExpressionEvaluationException(this, e);
		}
	}

	@Override
	protected void appendExpressionString(StringBuilder b)
	{
		b.append(operator);
		b.append(' ');
		child.appendExpressionString(b);
	}

	@Override
	protected void appendValuePattern(StringBuilder builder)
	{
		RegExpBuilder.appendTypeExpr(builder, getGenericType());
	}

	//==========================================================================
	// ====
	// === Operators Implementation
	//==========================================================================
	// ====
	public interface Operator
	{
		public boolean checkType(ITypeConverter converter, Class<?> t);

		public Class<?> getType(Class<?> t);

		public Object apply(ITypeConverter converter, Object o) throws Exception;

	}

	public static final Operator EMPTY = new Operator() {
		public boolean checkType(ITypeConverter converter, Class<?> t)
		{
// no: supports all type			return converter.isConvertible(t, Boolean.class);
			return true;
		}

		public Class<?> getType(Class<?> t)
		{
			return Boolean.class;
		}

		public Object apply(ITypeConverter converter, Object o) throws Exception
		{
			return BooleanOperations.empty(converter, o);
		}

		@Override
		public String toString()
		{
			return "empty";
		}
	};

	public static final Operator NEG = new Operator() {
		public boolean checkType(ITypeConverter converter, Class<?> t)
		{
			return NumberOperations.checkType(converter, t);
		}

		public Class<?> getType(Class<?> t)
		{
			return t;
		}

		public Object apply(ITypeConverter converter, Object o) throws Exception
		{
			return NumberOperations.neg(converter, o);
		}

		@Override
		public String toString()
		{
			return "-";
		}
	};

	public static final Operator NOT = new Operator() {
		public boolean checkType(ITypeConverter converter, Class<?> t)
		{
			return converter.isConvertible(t, Boolean.class);
		}

		public Class<?> getType(Class<?> t)
		{
			return Boolean.class;
		}

		public Object apply(ITypeConverter converter, Object o) throws Exception
		{
			return BooleanOperations.not(converter, o);
		}

		@Override
		public String toString()
		{
			return "!";
		}
	};
}
