/*
 * Copyright 2006, 2007 Odysseus Software GmbH
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.ow2.opensuit.cel.util;

import java.lang.reflect.Array;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.Calendar;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

import org.ow2.opensuit.cel.ITypeConverter;


/**
 * Type Conversions as described in EL 2.1 specification (section 1.17).
 */
public class StrictTypeConverter implements ITypeConverter
{
	private static final long serialVersionUID = 1L;

	protected static final Long LONG_ZERO = Long.valueOf(0L);
	protected static final Map<Class<?>, Class<?>> WRAPPER_TYPES;

	static
	{
		HashMap<Class<?>, Class<?>> wrapperTypes = new HashMap<Class<?>, Class<?>>();
		wrapperTypes = new HashMap<Class<?>, Class<?>>();
		wrapperTypes.put(Boolean.TYPE, Boolean.class);
		wrapperTypes.put(Character.TYPE, Character.class);
		wrapperTypes.put(Byte.TYPE, Byte.class);
		wrapperTypes.put(Short.TYPE, Short.class);
		wrapperTypes.put(Integer.TYPE, Integer.class);
		wrapperTypes.put(Long.TYPE, Long.class);
		wrapperTypes.put(Float.TYPE, Float.class);
		wrapperTypes.put(Double.TYPE, Double.class);
		WRAPPER_TYPES = Collections.unmodifiableMap(wrapperTypes);
	}
	/*
	private boolean isIntegerType(Class type)
	{
		return type == Short.class ||
		type == Integer.class ||
		type == Long.class ||
		BigInteger.class.isAssignableFrom(type);
	}
	private boolean isDecimalType(Class type)
	{
		return type == Float.class ||
			type == Double.class ||
			BigDecimal.class.isAssignableFrom(type);
	}
	*/
	private Boolean coerceToBoolean(Object value) throws ConversionError
	{
		if (value == null)
		{
			return Boolean.FALSE;
		}
		if (value instanceof Boolean)
		{
			return (Boolean) value;
		}
		if (value instanceof String)
		{
			// NO return Boolean.valueOf((String)value);
			return ((String) value).length() > 0;
		}
		if (value instanceof Number)
		{
			return ((Number) value).floatValue() != 0.0f;
		}
		throw new ConversionError(String.format("Cannot coerce from %s to %s", value.getClass(), Boolean.class));
	}

	private Character coerceToCharacter(Object value) throws ConversionError
	{
		if (value == null || "".equals(value))
		{
			return Character.valueOf((char) 0);
		}
		if (value instanceof Character)
		{
			return (Character) value;
		}
		if (value instanceof Number)
		{
			return Character.valueOf((char) ((Number) value).shortValue());
		}
		if (value instanceof String)
		{
			return Character.valueOf(((String) value).charAt(0));
		}
		throw new ConversionError(String.format("Cannot coerce from %s to %s", value.getClass(), Character.class));
	}

	/*
	 * protected Number coerceStringToNumber(String value, Class<? extends
	 * Number> type) { try { if (type == BigDecimal.class) { return new
	 * BigDecimal(value); } if (type == BigInteger.class) { return new
	 * BigInteger(value); } if (type == Byte.class) { return
	 * Byte.valueOf(value); } if (type == Short.class) { return
	 * Short.valueOf(value); } if (type == Integer.class) { return
	 * Integer.valueOf(value); } if (type == Long.class) { return
	 * Long.valueOf(value); } if (type == Float.class) { return
	 * Float.valueOf(value); } if (type == Double.class) { return
	 * Double.valueOf(value); } } catch (NumberFormatException e) { throw new
	 * ELException(LocalMessages.get("error.coerce.value", value, type)); }
	 * throw new ELException(String.format("Cannot coerce from %s to %s",
	 * String.class, type)); }
	 */

	private Number coerceNumberToNumber(Number value, Class<? extends Number> type) throws ConversionError
	{
		if (type.isInstance(value))
		{
			return value;
		}
		if (type == BigInteger.class)
		{
			if (value instanceof BigDecimal)
			{
				return ((BigDecimal) value).toBigInteger();
			}
			return BigInteger.valueOf((value).longValue());
		}
		if (type == BigDecimal.class)
		{
			if (value instanceof BigInteger)
			{
				return new BigDecimal((BigInteger) value);
			}
			return new BigDecimal(value.doubleValue());
		}
		if (type == Byte.class)
		{
			// TODO: throw exception if value out of bounds
			return Byte.valueOf(value.byteValue());
		}
		if (type == Short.class)
		{
			// TODO: throw exception if value out of bounds
			return Short.valueOf(value.shortValue());
		}
		if (type == Integer.class)
		{
			// TODO: throw exception if value out of bounds
			return Integer.valueOf(value.intValue());
		}
		if (type == Long.class)
		{
			// TODO: throw exception if value out of bounds
			return Long.valueOf(value.longValue());
		}
		if (type == Float.class)
		{
			// TODO: throw exception if value out of bounds
			return Float.valueOf(value.floatValue());
		}
		if (type == Double.class)
		{
			// TODO: throw exception if value out of bounds
			return Double.valueOf(value.doubleValue());
		}
		throw new ConversionError(String.format("Cannot coerce from %s to %s", value.getClass(), type));
	}

	@SuppressWarnings("unchecked")
	private <T extends Number> T coerceToNumber(Object value, Class<T> type) throws ConversionError
	{
		if (value == null)
		{
			return (T) coerceNumberToNumber(LONG_ZERO, type);
		}
		if (value instanceof Number)
		{
			return (T) coerceNumberToNumber((Number) value, type);
		}
		throw new ConversionError(String.format("Cannot coerce from %s to %s", value.getClass(), type));
	}

	private String coerceToString(Object value)
	{
		return coerceToString(value, false);
	}

	private String coerceToString(Object value, boolean inList)
	{
		if (value == null)
		{
			return "";
		}
		if (value instanceof String)
		{
			return (String) value;
		}
		if (value instanceof Enum<?>)
		{
			return ((Enum<?>) value).name();
		}
		if (value.getClass().isArray())
		{
			int l = Array.getLength(value);
			
			if (l == 0){
				return "";
			}
			
			StringBuffer array = new StringBuffer();
			
			if (inList){
				array.append('(');
			}
			
			for (int i = 0; i < l; i++)
			{
				if (i > 0){
					array.append(", ");
				}
				
				array.append(coerceToString(Array.get(value, i), true));
			}
			
			if (inList){
				array.append(')');
			}
			
			return array.toString();
		}
		if (value instanceof Collection<?>)
		{
			Collection<?> coll = (Collection<?>) value;
			int l = coll.size();
			
			if (l == 0){
				return "";
			}
			
			StringBuffer array = new StringBuffer();
			
			if (inList){
				array.append('(');
			}
			
			int i = 0;
			for (Object o : coll)
			{
				if (i > 0){
					array.append(", ");
				}
				
				array.append(coerceToString(o, true));
				i++;
			}
			
			if (inList){
				array.append(')');
			}
			
			return array.toString();
		}
		return value.toString();
	}

	private Date coerceToDate(Object value) throws ConversionError
	{
		if (value == null)
		{
			return null;
		}
		if (value instanceof Date)
		{
			return (Date) value;
		}
		if (value instanceof Calendar)
		{
			return ((Calendar) value).getTime();
		}
		throw new ConversionError(String.format("Cannot coerce from %s to %s", value.getClass(), Date.class));
	}

	private Calendar coerceToCalendar(Object value) throws ConversionError
	{
		if (value == null)
		{
			return null;
		}
		if (value instanceof Calendar)
		{
			return (Calendar) value;
		}
		if (value instanceof Date)
		{
			Calendar cal = Calendar.getInstance();
			cal.setTime((Date) value);
			return cal;
		}
		throw new ConversionError(String.format("Cannot coerce from %s to %s", value.getClass(), Calendar.class));
	}

	@SuppressWarnings("unchecked")
	private <T extends Enum<T>> T coerceToEnum(Object value, Class<T> type) throws ConversionError
	{
		if (value == null)
		{
			return null;
		}
		if (type.isInstance(value))
		{
			return (T) value;
		}
		if (value instanceof String)
		{
			try
			{
				return Enum.valueOf(type, (String) value);
			}
			catch (IllegalArgumentException e)
			{
				throw new ConversionError(String.format("Cannot coerce '%s' to %s", value, type));
			}
		}
		throw new ConversionError(String.format("Cannot coerce from %s to %s", value.getClass(), type));
	}

	/*
	 * protected Object coerceStringToType(String value, Class<?> type) {
	 * PropertyEditor editor = PropertyEditorManager.findEditor(type); if
	 * (editor == null) { if ("".equals(value)) { return null; } throw new
	 * ELException(String.format("Cannot coerce from %s to %s", String.class,
	 * type)); } else { if ("".equals(value)) { try { editor.setAsText(value); }
	 * catch (IllegalArgumentException e) { return null; } } else { try {
	 * editor.setAsText(value); } catch (IllegalArgumentException e) { throw new
	 * ELException(LocalMessages.get("error.coerce.value", value, type)); } }
	 * return editor.getValue(); } }
	 */

	public boolean isConvertible(Class<?> sourceType, Class<?> targetType)
	{
		if (targetType == null || sourceType == null){
			return false;
		}
		
		if (targetType == Object.class){
			return true;
		}
		
		if (targetType == String.class){
			// any object can be changed into a string
			return true;
		}

		if (sourceType.isPrimitive()){
			sourceType = WRAPPER_TYPES.get(sourceType);
		}
		
		if (targetType.isPrimitive()){
			targetType = WRAPPER_TYPES.get(targetType);
		}

		if (targetType.isAssignableFrom(sourceType)){
			// no coercion required
			return true;
		}
		
		// TODO: strict conversion means only integer types are compatible, and decimal types
		if (Number.class.isAssignableFrom(targetType)){
			// --- source type must be a Number too
			return Number.class.isAssignableFrom(sourceType);
		}
		
		/*
		 * ??? if (type == Character.class) { return coerceToCharacter(value); }
		 */
		if (targetType == Boolean.class)
		{
			// any object can be changed into a boolean
			// no: strict means we expect a boolean // return true;
			return sourceType == Boolean.class || sourceType == Boolean.TYPE;
		}
		
		if (Date.class.isAssignableFrom(targetType)){
			// --- Calendar is compatible with Date
			return Calendar.class.isAssignableFrom(sourceType);
		}

		if (Calendar.class.isAssignableFrom(targetType)){
			// --- Calendar is compatible with Date
			return Date.class.isAssignableFrom(sourceType);
		}

		/*
		 * already tested if (Enum.class.isAssignableFrom(targetType)) { return
		 * targetType.isAssignableFrom(sourceType); }
		 */
		return false;
	}

	@SuppressWarnings("unchecked")
	public <T> T convert(Object value, Class<T> type) throws ConversionError
	{
		if (value == null )
		{
			if (!type.isPrimitive()){
				return null;
			}
			// else continue --> we will convert null to the default value
		}

		if (type == Object.class){
			return (T)value;
		}

		if (type.isInstance(value)){
			return (T)value;
		}
		
		if (type == String.class){
			return (T)coerceToString(value);
		}
		
		if (type.isPrimitive()){
			type = (Class<T>)WRAPPER_TYPES.get(type);
		}
		
		if (Number.class.isAssignableFrom(type)){
			return (T)coerceToNumber(value, (Class<? extends Number>) type);
		}
		
		if (type == Character.class){
			return (T)coerceToCharacter(value);
		}
		
		if (type == Boolean.class){
			return (T)coerceToBoolean(value);
		}
		
		if (Enum.class.isAssignableFrom(type)){
			return (T)coerceToEnum(value, (Class<? extends Enum>) type);
		}
		
		if (Date.class.isAssignableFrom(type)){
			return (T)coerceToDate(value);
		}
		
		if (Calendar.class.isAssignableFrom(type)){
			return (T)coerceToCalendar(value);
		}
		
		throw new ConversionError(String.format("Cannot coerce from %s to %s", value.getClass(), type));
	}
	
	public static void main(String[] args) throws ConversionError
	{
		StrictTypeConverter c = new StrictTypeConverter();
		boolean val = c.convert(null, Boolean.TYPE);
		System.out.println("value: "+val);
	}
}
