/**
 * Open SUIT - Simple User Interface Toolkit
 * 
 * Copyright (c) 2009 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package org.ow2.opensuit.xml.chart;

import java.awt.Color;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import javax.servlet.http.HttpServletRequest;

import org.jfree.chart.axis.CategoryAxis;
import org.jfree.chart.axis.CategoryAxis3D;
import org.jfree.chart.axis.NumberAxis;
import org.jfree.chart.axis.NumberAxis3D;
import org.jfree.chart.axis.ValueAxis;
import org.jfree.chart.labels.CategoryToolTipGenerator;
import org.jfree.chart.labels.ItemLabelAnchor;
import org.jfree.chart.labels.ItemLabelPosition;
import org.jfree.chart.labels.StandardCategoryToolTipGenerator;
import org.jfree.chart.plot.CategoryPlot;
import org.jfree.chart.plot.Marker;
import org.jfree.chart.plot.Plot;
import org.jfree.chart.plot.PlotOrientation;
import org.jfree.chart.plot.ValueMarker;
import org.jfree.chart.renderer.category.AreaRenderer;
import org.jfree.chart.renderer.category.BarRenderer;
import org.jfree.chart.renderer.category.BarRenderer3D;
import org.jfree.chart.renderer.category.CategoryItemRenderer;
import org.jfree.chart.renderer.category.LineAndShapeRenderer;
import org.jfree.chart.renderer.category.LineRenderer3D;
import org.jfree.chart.renderer.category.StackedBarRenderer;
import org.jfree.chart.renderer.category.StackedBarRenderer3D;
import org.jfree.chart.renderer.category.WaterfallBarRenderer;
import org.jfree.chart.urls.CategoryURLGenerator;
import org.jfree.data.category.CategoryDataset;
import org.jfree.data.category.DefaultCategoryDataset;
import org.jfree.ui.Layer;
import org.jfree.ui.TextAnchor;
import org.jfree.util.SortOrder;
import org.ow2.opensuit.core.session.OpenSuitSession;
import org.ow2.opensuit.core.util.ReflectionHelper;
import org.ow2.opensuit.xml.base.action.IAction;
import org.ow2.opensuit.xml.base.binding.Expression;
import org.ow2.opensuit.xml.interfaces.IBeanProvider;
import org.ow2.opensuit.xmlmap.annotations.XmlAttribute;
import org.ow2.opensuit.xmlmap.annotations.XmlChild;
import org.ow2.opensuit.xmlmap.annotations.XmlDoc;
import org.ow2.opensuit.xmlmap.annotations.XmlElement;
import org.ow2.opensuit.xmlmap.interfaces.IInitializationSupport;
import org.ow2.opensuit.xmlmap.interfaces.IInstantiationContext;

@XmlElement
public class CategoryChart extends BaseChart implements IBeanProvider
{
	@XmlDoc("The name of the category contextual bean.<br>" + 
			"This bean is available at render-time only.<br>" + 
			"Default: $category.")
	@XmlAttribute(name = "CategoryVar", required = false)
	private String categoryVar = "$category";

	@XmlDoc("The name of the serie contextual bean.<br>" + 
			"This bean is available at render-time only.<br>" + 
			"Default: $serie.")
	@XmlAttribute(name = "SerieVar", required = false)
	private String serieVar = "$serie";

	@XmlDoc("The name of the value contextual bean.<br>" + 
			"This bean is available at render-time only.<br>" + 
			"Default: $value.")
	@XmlAttribute(name = "ValueVar", required = false)
	private String valueVar = "$value";

	@XmlDoc("Determines the chart orientation.<br>" + 
			"Default: horizontal.")
	@XmlAttribute(name = "Orientation", required = false)
	private Orientation orientation = Orientation.Horizontal;

	@XmlDoc("Determines the chart style.<br>" + 
			"Default: Bar.")
	@XmlAttribute(name = "Style", required = false)
	private CategoryChartStyle style = CategoryChartStyle.Bar;

	@XmlDoc("Defines the category axis label.")
	@XmlChild(name = "CategoryAxisLabel")
	protected Expression categoryAxisLabel;

	@XmlDoc("Defines the value axis label.")
	@XmlChild(name = "ValueAxisLabel")
	protected Expression valueAxisLabel;

	@XmlDoc("Gives a vector of items representing the categories.")
	@XmlChild(name = "Categories")
	protected Expression categories;

	@XmlDoc("Gives a single category name.<br>" + 
			"Supported contextual beans: $category")
	@XmlChild(name = "CategoryName")
	protected Expression categoryName;

	@XmlDoc("Gives a vector of items representing the series.")
	@XmlChild(name = "Series")
	protected Expression series;

	@XmlDoc("Gives a single serie name.<br>" + 
			"Supported contextual beans: $serie")
	@XmlChild(name = "SerieName")
	protected Expression serieName;

	@XmlDoc("Gives a value for a given category and a serie.<br>" +
			"Supported contextual beans: $category, $serie")
	@XmlChild(name = "Value")
	protected Expression value;

	@XmlDoc("Defines the tooltip for a given chart value.<br>" +
			"Supported contextual beans: $category, $serie, $value")
	@XmlChild(name = "ValueTooltip", required = false)
	protected Expression valueTooltip;

	@XmlDoc("Defines the action to perform when a given chart value is clicked.<br>" +
			"Supported contextual beans: $category, $serie, $value")
	@XmlChild(name = "OnClickValue", required = false)
	protected IAction onClickValue;

	private Type categoryType;
	private Class<?> categoryClass;

	private Type serieType;
	private Class<?> serieClass;

	private Type valueType;
	private Class<?> valueClass;

	public void initialize(IInitializationSupport initSupport, IInstantiationContext instContext)
	{
		super.initialize(initSupport, instContext);

		if (initSupport.initialize(categories))
		{
			categoryClass = ReflectionHelper.getVectorElementClass(categories.getGenericType());
			if (categoryClass == null){
				initSupport.addValidationMessage(this, "Categories", IInitializationSupport.ERROR, "Expression 'Categories' must return vector data.");
			} else {
				categoryType = ReflectionHelper.getVectorElementType(categories.getGenericType());
			}
		}
		if (initSupport.initialize(series))
		{
			serieClass = ReflectionHelper.getVectorElementClass(series.getGenericType());
			if (serieClass == null){
				initSupport.addValidationMessage(this, "Series", IInitializationSupport.ERROR, "Expression 'Series' must return vector data.");
			} else {
				serieType = ReflectionHelper.getVectorElementType(series.getGenericType());
			}
		}
		if (initSupport.initialize(value))
		{
			if (!value.isConvertible(Number.class))
			{
				initSupport.addValidationMessage(this, "Value", IInitializationSupport.ERROR, "Expression 'Value' must return a Number.");
			}
			else
			{
				valueClass = value.getType();
				valueType = value.getGenericType();
			}
		}
	}

	@Override
	public Plot makePlot(HttpServletRequest request) throws Exception
	{
		// --- 1: make dataset
		Collection<?> categoriesObjs = ReflectionHelper.obj2Collection(categories.invoke(request));
		
		if (categoriesObjs == null){
			return null;
		}

		Collection<?> seriesObjs = ReflectionHelper.obj2Collection(series.invoke(request));
		
		if (seriesObjs == null){
			return null;
		}

		String categoryAxisLabel = this.categoryAxisLabel.invoke(request, String.class);
		String valueAxisLabel = this.valueAxisLabel.invoke(request, String.class);
		PlotOrientation orientation = this.orientation.getPlotOrientation();
		/*
		 * // --- retrieve names ArrayList<String> categoryNames = new
		 * ArrayList<String>(); for(Object category : categoriesObjs) {
		 * request.setAttribute(categoryVar, category); String name =
		 * categoryName.invoke(request, String.class); categoryNames.add(name);
		 * } ArrayList<String> serieNames = new ArrayList<String>(); for(Object
		 * serie : seriesObjs) { request.setAttribute(serieVar, serie); String
		 * name = serieName.invoke(request, String.class); serieNames.add(name);
		 * }
		 */
		// --- build the dataset
		List<Key> categoryKeys = new ArrayList<Key>();
		int i=0;
		for (Object category : categoriesObjs)
		{
			request.setAttribute(categoryVar, category);
			Key catKey = new Key();
			catKey.item = category;
			catKey.name = categoryName.invoke(request, String.class);
			catKey.index = i++;
			categoryKeys.add(catKey);
		}
		
		List<Key> serieKeys = new ArrayList<Key>();
		i=0;
		for (Object serie : seriesObjs)
		{
			request.setAttribute(serieVar, serie);
			Key serKey = new Key();
			serKey.item = serie;
			serKey.name = serieName.invoke(request, String.class);
			serKey.index = i++;
			serieKeys.add(serKey);
		}
		
		DefaultCategoryDataset dataset = new DefaultCategoryDataset();
		for (Key catKey : categoryKeys)
		{
			request.setAttribute(categoryVar, catKey.item);
			for (Key serKey : serieKeys)
			{
				request.setAttribute(serieVar, serKey.item);

				Number val = value.invoke(request, Number.class);
				dataset.setValue(val, serKey, catKey);
			}
		}
		// Area, Bar, Bar3D, Line, Line3D, StackedBar, StackedBar3D, Waterfall

		// --- 2: make plot
		CategoryPlot plot = null;
		if (style == CategoryChartStyle.Area)
		{
			// return ChartFactory.createAreaChart(title, catAxisLabel,
			// valAxisLabel, dataset, orientation.getPlotOrientation(),
			// showLegend, renderTooltips, renderUrls);
			CategoryAxis categoryAxis = new CategoryAxis(categoryAxisLabel);
			categoryAxis.setCategoryMargin(0.0);
			ValueAxis valueAxis = new NumberAxis(valueAxisLabel);
			AreaRenderer renderer = new AreaRenderer();
			plot = new CategoryPlot(dataset, categoryAxis, valueAxis, renderer);
			plot.setOrientation(orientation);
		}
		else if (style == CategoryChartStyle.Bar)
		{
			// return ChartFactory.createBarChart(title, catAxisLabel,
			// valAxisLabel, dataset, orientation.getPlotOrientation(),
			// showLegend, renderTooltips, renderUrls);
			CategoryAxis categoryAxis = new CategoryAxis(categoryAxisLabel);
			ValueAxis valueAxis = new NumberAxis(valueAxisLabel);

			BarRenderer renderer = new BarRenderer();
			if (orientation == PlotOrientation.HORIZONTAL)
			{
				ItemLabelPosition position1 = new ItemLabelPosition(ItemLabelAnchor.OUTSIDE3, TextAnchor.CENTER_LEFT);
				renderer.setBasePositiveItemLabelPosition(position1);
				ItemLabelPosition position2 = new ItemLabelPosition(ItemLabelAnchor.OUTSIDE9, TextAnchor.CENTER_RIGHT);
				renderer.setBaseNegativeItemLabelPosition(position2);
			}
			else if (orientation == PlotOrientation.VERTICAL)
			{
				ItemLabelPosition position1 = new ItemLabelPosition(ItemLabelAnchor.OUTSIDE12, TextAnchor.BOTTOM_CENTER);
				renderer.setBasePositiveItemLabelPosition(position1);
				ItemLabelPosition position2 = new ItemLabelPosition(ItemLabelAnchor.OUTSIDE6, TextAnchor.TOP_CENTER);
				renderer.setBaseNegativeItemLabelPosition(position2);
			}

			plot = new CategoryPlot(dataset, categoryAxis, valueAxis, renderer);
			plot.setOrientation(orientation);
		}
		else if (style == CategoryChartStyle.Bar3D)
		{
			// valAxisLabel, dataset, orientation.getPlotOrientation(),
			// showLegend, renderTooltips, renderUrls);
			CategoryAxis categoryAxis = new CategoryAxis3D(categoryAxisLabel);
			ValueAxis valueAxis = new NumberAxis3D(valueAxisLabel);

			BarRenderer3D renderer = new BarRenderer3D();
			plot = new CategoryPlot(dataset, categoryAxis, valueAxis, renderer);
			plot.setOrientation(orientation);
			if (orientation == PlotOrientation.HORIZONTAL)
			{
				// change rendering order to ensure that bar overlapping is the
				// right way around
				plot.setRowRenderingOrder(SortOrder.DESCENDING);
				plot.setColumnRenderingOrder(SortOrder.DESCENDING);
			}
			plot.setForegroundAlpha(0.75f);
		}
		else if (style == CategoryChartStyle.Line)
		{
			// return ChartFactory.createLineChart(title, catAxisLabel,
			// valAxisLabel, dataset, orientation.getPlotOrientation(),
			// showLegend, renderTooltips, renderUrls);
			CategoryAxis categoryAxis = new CategoryAxis(categoryAxisLabel);
			ValueAxis valueAxis = new NumberAxis(valueAxisLabel);

			LineAndShapeRenderer renderer = new LineAndShapeRenderer(true, false);
			plot = new CategoryPlot(dataset, categoryAxis, valueAxis, renderer);
			plot.setOrientation(orientation);
		}
		else if (style == CategoryChartStyle.Line3D)
		{
			// return ChartFactory.createLineChart3D(title, catAxisLabel,
			// valAxisLabel, dataset, orientation.getPlotOrientation(),
			// showLegend, renderTooltips, renderUrls);
			CategoryAxis categoryAxis = new CategoryAxis3D(categoryAxisLabel);
			ValueAxis valueAxis = new NumberAxis3D(valueAxisLabel);
			LineRenderer3D renderer = new LineRenderer3D();
			plot = new CategoryPlot(dataset, categoryAxis, valueAxis, renderer);
			plot.setOrientation(orientation);
		}
		else if (style == CategoryChartStyle.StackedBar)
		{
			// return ChartFactory.createStackedBarChart(title, catAxisLabel,
			// valAxisLabel, dataset, orientation.getPlotOrientation(),
			// showLegend, renderTooltips, renderUrls);
			CategoryAxis categoryAxis = new CategoryAxis(categoryAxisLabel);
			ValueAxis valueAxis = new NumberAxis(valueAxisLabel);

			StackedBarRenderer renderer = new StackedBarRenderer();

			plot = new CategoryPlot(dataset, categoryAxis, valueAxis, renderer);
			plot.setOrientation(orientation);
		}
		else if (style == CategoryChartStyle.StackedBar3D)
		{
			// return ChartFactory.createStackedBarChart3D(title, catAxisLabel,
			// valAxisLabel, dataset, orientation.getPlotOrientation(),
			// showLegend, renderTooltips, renderUrls);
			CategoryAxis categoryAxis = new CategoryAxis3D(categoryAxisLabel);
			ValueAxis valueAxis = new NumberAxis3D(valueAxisLabel);

			// create the renderer...
			CategoryItemRenderer renderer = new StackedBarRenderer3D();

			// create the plot...
			plot = new CategoryPlot(dataset, categoryAxis, valueAxis, renderer);
			plot.setOrientation(orientation);
			if (orientation == PlotOrientation.HORIZONTAL)
			{
				// change rendering order to ensure that bar overlapping is the
				// right way around
				plot.setColumnRenderingOrder(SortOrder.DESCENDING);
			}
		}
		else
		// Waterfall
		{
			// return ChartFactory.createWaterfallChart(title, catAxisLabel,
			// valAxisLabel, dataset, orientation.getPlotOrientation(),
			// showLegend, renderTooltips, renderUrls);
			CategoryAxis categoryAxis = new CategoryAxis(categoryAxisLabel);
			categoryAxis.setCategoryMargin(0.0);

			ValueAxis valueAxis = new NumberAxis(valueAxisLabel);

			WaterfallBarRenderer renderer = new WaterfallBarRenderer();
			if (orientation == PlotOrientation.HORIZONTAL)
			{
				ItemLabelPosition position = new ItemLabelPosition(ItemLabelAnchor.CENTER, TextAnchor.CENTER, TextAnchor.CENTER, Math.PI / 2.0);
				renderer.setBasePositiveItemLabelPosition(position);
				renderer.setBaseNegativeItemLabelPosition(position);
			}
			else if (orientation == PlotOrientation.VERTICAL)
			{
				ItemLabelPosition position = new ItemLabelPosition(ItemLabelAnchor.CENTER, TextAnchor.CENTER, TextAnchor.CENTER, 0.0);
				renderer.setBasePositiveItemLabelPosition(position);
				renderer.setBaseNegativeItemLabelPosition(position);
			}

			plot = new CategoryPlot(dataset, categoryAxis, valueAxis, renderer);
			plot.clearRangeMarkers();
			Marker baseline = new ValueMarker(0.0);
			baseline.setPaint(Color.black);
			plot.addRangeMarker(baseline, Layer.FOREGROUND);
			plot.setOrientation(orientation);
		}

		// --- 3: apply generators
		// --- add the tooltip generator
		if (valueTooltip != null)
		{
			plot.getRenderer().setBaseToolTipGenerator(new TooltipGenerator());
		}
		else
		{
			// --- apply default tooltip generator
			plot.getRenderer().setBaseToolTipGenerator(new StandardCategoryToolTipGenerator());
		}

		// --- add the url generator
		if (onClickValue != null){
			plot.getRenderer().setBaseItemURLGenerator(new URLGenerator());
		}

		return plot;
	}

	// ==========================================================================
	// === Key
	// ==========================================================================
	private static class Key implements Comparable<Key>
	{
		private int index;
		private String name;
		private Object item;
		
		public int compareTo(Key o)
		{
			return index - o.index;
		}
		@Override
		public String toString()
		{
			return name;
		}
		@Override
		public boolean equals(Object obj)
		{
			if(obj == null || !(obj instanceof Key)){
				return false;
			}
			return item.equals(((Key)obj).item);
		}
		@Override
		public int hashCode()
		{
			return item.hashCode();
		}
	}
	// ==========================================================================
	// === TooltipGenerator
	// ==========================================================================
	private class TooltipGenerator implements CategoryToolTipGenerator
	{
		public String generateToolTip(CategoryDataset dataset, int series, int category)
		{
			HttpServletRequest request = OpenSuitSession.getCurrentRequest();

			Key catKey = (Key) dataset.getColumnKey(category);
			Key serieKey = (Key) dataset.getRowKey(series);
			Number value = dataset.getValue(series, category);

			// --- set contextual beans
			request.setAttribute(categoryVar, catKey.item);
			request.setAttribute(serieVar, serieKey.item);
			request.setAttribute(valueVar, value);

			// --- then eval tooltip
			try
			{
				return valueTooltip.invoke(request, String.class);
			}
			catch (Exception e)
			{
				logger.error("Error while evaluating item tooltip.", e);
				return null;
			}
		}
		@Override
		public boolean equals(Object obj)
		{
			logger.debug("TooltipGenerator.equals()");
			return obj != null && (obj instanceof TooltipGenerator);
		}
	}

	//==========================================================================
	// === UrlGenerator
	//==========================================================================
	private class URLGenerator implements CategoryURLGenerator
	{
		public String generateURL(CategoryDataset dataset, int series, int category)
		{
			HttpServletRequest request = OpenSuitSession.getCurrentRequest();

			Key catKey = (Key) dataset.getColumnKey(category);
			Key serieKey = (Key) dataset.getRowKey(series);
			Number value = dataset.getValue(series, category);

			// --- set contextual beans
			request.setAttribute(categoryVar, catKey.item);
			request.setAttribute(serieVar, serieKey.item);
			request.setAttribute(valueVar, value);

			// --- then eval url
			try
			{
				return onClickValue.getURL(request, true);
			}
			catch (Exception e)
			{
				logger.error("Error while evaluating onclick action.", e);
				return null;
			}
		}
		@Override
		public boolean equals(Object obj)
		{
			logger.debug("URLGenerator.equals()");
			return obj != null && (obj instanceof URLGenerator);
		}
	}

	//==========================================================================
	// === IBeanProvider
	//==========================================================================
	@Override
	public Class<?> getBeanType(String iName) throws UnresolvedBeanError
	{
		if (categoryVar.equals(iName))
		{
			if (categoryClass == null){
				throw new UnresolvedBeanError();
			}
			
			return categoryClass;
		}
		else if (serieVar.equals(iName))
		{
			if (serieClass == null){
				throw new UnresolvedBeanError();
			}
			
			return serieClass;
		}
		else if (valueVar.equals(iName))
		{
			if (valueClass == null){
				throw new UnresolvedBeanError();
			}
			
			return valueClass;
		}
		return super.getBeanType(iName);
	}

	@Override
	public Type getBeanGenericType(String iName) throws UnresolvedBeanError
	{
		if (categoryVar.equals(iName))
		{
			if (categoryType == null){
				throw new UnresolvedBeanError();
			}
			
			return categoryType;
		}
		else if (serieVar.equals(iName))
		{
			if (serieType == null){
				throw new UnresolvedBeanError();
			}
			
			return serieType;
		}
		else if (valueVar.equals(iName))
		{
			if (valueType == null){
				throw new UnresolvedBeanError();
			}
			
			return valueType;
		}
		return super.getBeanGenericType(iName);
	}

	@Override
	public Object getBeanValue(HttpServletRequest iRequest, String iName) throws Exception
	{
		if (categoryVar.equals(iName)){
			return iRequest.getAttribute(iName);
		} else if (serieVar.equals(iName)){
			return iRequest.getAttribute(iName);
		} else if (valueVar.equals(iName)){
			return iRequest.getAttribute(iName);
		}
		return super.getBeanValue(iRequest, iName);
	}
	@Override
	public List<String> getBeanNames() {
		List<String> names = super.getBeanNames();
		names.add(categoryVar);
		names.add(serieVar);
		names.add(valueVar);
		return names;
	}
	@Override
	public String getBeanDescription(String name) {
		if(categoryVar.equals(name))
		{
			return "The &lt;CategoryChart&gt; category contextual bean.<br>" + 
				"This bean is available at render-time only.<br>" + 
				"Class: "+(categoryClass == null ? "unresolved!" : categoryClass.getName());
		}
		else if (serieVar.equals(name))
		{
			return "The &lt;CategoryChart&gt; series contextual bean.<br>" + 
			"This bean is available at render-time only.<br>" + 
			"Class: "+(serieClass == null ? "unresolved!" : serieClass.getName());
		}
		else if (valueVar.equals(name))
		{
			return "The &lt;CategoryChart&gt; value contextual bean.<br>" + 
			"This bean is available at render-time only.<br>" + 
			"Class: "+(valueClass == null ? "unresolved!" : valueClass.getName());
		}

		// TODO Auto-generated method stub
		return super.getBeanDescription(name);
	}
}
