/**
 * Open SUIT - Simple User Interface Toolkit
 * 
 * Copyright (c) 2009 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package org.ow2.opensuit.xml.chart;

import java.lang.reflect.Type;
import java.util.Collection;

import javax.servlet.http.HttpServletRequest;

import org.jfree.chart.labels.PieToolTipGenerator;
import org.jfree.chart.labels.StandardPieSectionLabelGenerator;
import org.jfree.chart.labels.StandardPieToolTipGenerator;
import org.jfree.chart.plot.PiePlot;
import org.jfree.chart.plot.PiePlot3D;
import org.jfree.chart.plot.RingPlot;
import org.jfree.chart.urls.PieURLGenerator;
import org.jfree.data.general.DefaultPieDataset;
import org.jfree.data.general.PieDataset;
import org.ow2.opensuit.core.session.OpenSuitSession;
import org.ow2.opensuit.core.util.ReflectionHelper;
import org.ow2.opensuit.xml.base.action.IAction;
import org.ow2.opensuit.xml.base.binding.Expression;
import org.ow2.opensuit.xml.interfaces.IBeanProvider;
import org.ow2.opensuit.xmlmap.annotations.XmlAttribute;
import org.ow2.opensuit.xmlmap.annotations.XmlChild;
import org.ow2.opensuit.xmlmap.annotations.XmlDoc;
import org.ow2.opensuit.xmlmap.annotations.XmlElement;
import org.ow2.opensuit.xmlmap.interfaces.IInitializationSupport;
import org.ow2.opensuit.xmlmap.interfaces.IInstantiationContext;

@XmlElement
public class PieChart extends BaseChart implements IBeanProvider
{
	@XmlDoc("The name of the item contextual bean.<br>" +
			"This bean is available at render-time only.<br>" +
			"Default: $item.")
	@XmlAttribute(name="IteratorVar", required=false)
	private String iteratorVar= "$item";
	
	@XmlDoc("The name of the value contextual bean.<br>" +
			"This bean is available at render-time only.<br>" +
			"Default: $value.")
	@XmlAttribute(name="ValueVar", required=false)
	private String valueVar= "$value";
	
	@XmlDoc("Determines the pie style.<br>" + 
			"Default: TwoD.")
	@XmlAttribute(name="Style", required=false)
	private PieChartStyle style= PieChartStyle.TwoD;
	
	@XmlDoc("Gives a vector of items representing pie sectors.")
	@XmlChild(name="Items")
	protected Expression items;
	
	@XmlDoc("Evaluates the name of a given pie item.<br>" + 
			"Supported contextual beans: $item")
	@XmlChild(name="ItemName")
	protected Expression itemName;
	
	@XmlDoc("Evaluates the value of a given pie item.<br>" + 
			"Supported contextual beans: $item")
	@XmlChild(name="ItemValue")
	protected Expression itemValue;
	
	@XmlDoc("Defines the tooltip for a given pie item.<br>" +
			"Supported contextual beans: $item, $value")
	@XmlChild(name="ItemTooltip", required=false)
	protected Expression itemTooltip;
	
	@XmlDoc("Defines the action to perform when a given pie item is clicked.<br>" +
			"Supported contextual beans: $item, $value")
	@XmlChild(name="OnClickItem", required=false)
	protected IAction onClickItem;
	
	private Type itemType;
	private Class<?> itemClass;
	
	private Type valueType;
	private Class<?> valueClass;
	
	public void initialize(IInitializationSupport initSupport, IInstantiationContext instContext)
	{
		super.initialize(initSupport, instContext);
		
		if(initSupport.initialize(items))
		{
			itemClass = ReflectionHelper.getVectorElementClass(items.getGenericType());
			if(itemClass == null)
			{
		    	initSupport.addValidationMessage(this, "Items", IInitializationSupport.ERROR, "Expression 'Items' must return vector data.");
			}
			else
			{
				itemType = ReflectionHelper.getVectorElementType(items.getGenericType());
				
				// --- check itemValue is a Number
				if(itemValue != null && initSupport.initialize(itemValue))
				{
					if(!itemValue.isConvertible(Number.class))
					{
				    	initSupport.addValidationMessage(this, "ItemValue", IInitializationSupport.ERROR, "Expression 'ItemValue' must return a Number.");
					}
					else
					{
						valueType = itemValue.getGenericType();
						valueClass = itemValue.getType();
					}
				}
			}
		}
	}
	@Override
	public PiePlot makePlot(HttpServletRequest request) throws Exception
	{
		// --- 1: make dataset
		Collection<?> itemObjs = ReflectionHelper.obj2Collection(items.invoke(request));
		if(itemObjs == null)
			return null;
		
		DefaultPieDataset dataset = new DefaultPieDataset();
		int i=0;
		for(Object item : itemObjs)
		{
			// --- set iterator var
			request.setAttribute(iteratorVar, item);
			Key key = new Key();
			key.item = item;
			key.name = itemName.invoke(request, String.class);
			key.index = i++;
			Number value = itemValue.invoke(request, Number.class);
			dataset.setValue(key, value);
		}
		
		// --- 2: make plot
		PiePlot plot = null;
		if(style == PieChartStyle.ThreeD)
		{
			plot = new PiePlot3D(dataset);
		}
		else if(style == PieChartStyle.Ring)
		{
	        plot = new RingPlot(dataset);
		}
		else // PieChart
		{
	        plot = new PiePlot(dataset);
		}
		
		// --- 3: apply generators
        plot.setLabelGenerator(new StandardPieSectionLabelGenerator());

		// --- add the tooltip generator
		if(itemTooltip != null)
		{
			plot.setToolTipGenerator(new TooltipGenerator());
		}
		else
		{
			// --- apply default tooltip generator
			OpenSuitSession session = OpenSuitSession.getSession(request);
			plot.setToolTipGenerator(new StandardPieToolTipGenerator(session.getLocale()));
		}
		
		// --- add the url generator
		if(onClickItem != null)
			plot.setURLGenerator(new URLGenerator());
		
		return plot;
	}
	// =================================================================================
	// === PieKey
	// =================================================================================
	private static class Key implements Comparable<Key>
	{
		private String name;
		private Object item;
		private int index;
		
		public int compareTo(Key o)
		{
			return index - o.index;
		}
		@Override
		public String toString()
		{
			return name;
		}
		@Override
		public boolean equals(Object obj)
		{
			if(obj == null || !(obj instanceof Key))
				return false;
			return item.equals(((Key)obj).item);
		}
		@Override
		public int hashCode()
		{
			return item.hashCode();
		}
	}
	// =================================================================================
	// === TooltipGenerator
	// =================================================================================
	private class TooltipGenerator implements PieToolTipGenerator
	{
		public String generateToolTip(PieDataset dataset, Comparable key)
		{
			HttpServletRequest request = OpenSuitSession.getCurrentRequest();
			Key k = (Key)key;
			// --- set iterator var
			request.setAttribute(iteratorVar, k.item);
			request.setAttribute(valueVar, dataset.getValue(key));
			// --- then eval tooltip
			try
			{
				return itemTooltip.invoke(request, String.class);
			}
			catch (Exception e)
			{
				logger.error("Error while evaluating item tooltip.", e);
				return null;
			}
		}
	}
	// =================================================================================
	// === UrlGenerator
	// =================================================================================
	private class URLGenerator implements PieURLGenerator
	{
		public String generateURL(PieDataset dataset, Comparable key, int pieIndex)
		{
			HttpServletRequest request = OpenSuitSession.getCurrentRequest();
			Key k = (Key)key;
			// --- set contextual beans
			request.setAttribute(iteratorVar, k.item);
			request.setAttribute(valueVar, dataset.getValue(key));
			// --- then eval action
			try
			{
				return onClickItem.getURL(request, true);
			}
			catch (Exception e)
			{
				logger.error("Error while evaluating onclick action.", e);
				return null;
			}
		}
	}
	// =================================================================================
	// === IBeanProvider
	// =================================================================================
	public Class<?> getBeanType(String iName) throws UnresolvedBeanError
	{
		if(iteratorVar.equals(iName))
		{
			if(itemClass == null) throw new UnresolvedBeanError();
			return itemClass;
		}
		else if(valueVar.equals(iName))
		{
			if(valueClass == null) throw new UnresolvedBeanError();
			return valueClass;
		}
		return super.getBeanType(iName);
	}
	public Type getBeanGenericType(String iName) throws UnresolvedBeanError
	{
		if(iteratorVar.equals(iName))
		{
			if(itemType == null) throw new UnresolvedBeanError();
			return itemType;
		}
		else if(valueVar.equals(iName))
		{
			if(valueType == null) throw new UnresolvedBeanError();
			return valueType;
		}
		return super.getBeanGenericType(iName);
	}
	public Object getBeanValue(HttpServletRequest iRequest, String iName) throws Exception
	{
		if(iteratorVar.equals(iName))
			return iRequest.getAttribute(iName);
		else if(valueVar.equals(iName))
			return iRequest.getAttribute(iName);
		return super.getBeanValue(iRequest, iName);
	}
}
