package org.ow2.opensuit.samples.movies_db.uibeans;

import java.util.ArrayList;
import java.util.List;

import org.ow2.opensuit.core.error.LocalizedError;
import org.ow2.opensuit.samples.movies_db.model.dataaccess.DataBaseError;
import org.ow2.opensuit.samples.movies_db.model.dataaccess.MoviesDataBase;
import org.ow2.opensuit.samples.movies_db.model.to.PersonTO;



/**
 * This bean holds a list of directors.
 * 
 * It is initilized either by loading all directors from the database, or
 * by searching directors by name.
 * 
 * <p>This version of the bean performs the search when loading, then
 * lets the framework manage sort and pagination.
 * This design may only be used if you're sure the amount of data is "reasonable"
 * and under control.
 * <p>See SearchMovieBean2 for a dynamic version of it (that have the model
 * layer manage sorting and pagination).
 */
public class ListOfDirectorsBean2
{
	// ==========================================================
	// === Attributes
	// ==========================================================
	private String name;
	private PersonTO[] directors;
	
	// ==========================================================
	// === Control Methods
	// ==========================================================
	/**
	 * Loads all directors from the database
	 */
	public void loadAll() throws LocalizedError
	{
		try
        {
	        directors = MoviesDataBase.getDataBase().getAllDirectors(null);
        }
        catch(DataBaseError e)
        {
			throw new LocalizedError("database_technical", new Object[]{e.getMessage()});
        }
	}
	/**
	 * Searches directors by name
	 */
	public String searchByName() throws LocalizedError
	{
		try
        {
	        directors = MoviesDataBase.getDataBase().searchDirectorByName(name, null);
        }
        catch(DataBaseError e)
        {
			throw new LocalizedError("database_technical", new Object[]{e.getMessage()});
        }
		
        if(directors == null || directors.length == 0){
			return "zero";
		}
		
		if(directors.length == 1){
			return "one";
		}
		
		return "several";
	}
	/**
	 * Deletes a director
	 * @param id
	 * @throws DataBaseError 
	 * @throws LocalizedError 
	 */
	public void delete(String id) throws DataBaseError, LocalizedError
	{
		// --- remove and reload all
		MoviesDataBase.getDataBase().removeDirector(Integer.parseInt(id));
		loadAll();
	}
	/**
	 * Resets the list
	 */
	public void reset()
	{
		directors = null;
		name = null;
	}
	/**
	 * Determines whether a list should be displayed
	 */
	public boolean hasList()
	{
		return directors != null;
	}
	/**
	 * Returns a list of suggested director names
	 * @param name
	 * @return
	 */
	public List<String> suggestDirectorNames(String name)
	{
		// --- suggest for at least one letter
		if(name == null){
			return null;
		}
		
		name = name.trim();
		
		if(name.length() == 0){
			return null;
		}
		
		try
		{
			PersonTO[] directors = MoviesDataBase.getDataBase().searchDirectorByName(name, null);
			List<String> suggestions = new ArrayList<String>(directors.length);
			
			for(PersonTO d : directors){
				suggestions.add(d.getName());
			}
			
			return suggestions;
		}
		catch (DataBaseError e)
		{
		}
		return null;
	}
	// ==========================================================
	// === Attributes Getters & Setters
	// ==========================================================
	public String getName()
    {
    	return name;
    }
	public void setName(String name)
    {
    	this.name = name;
    }
	public PersonTO[] getDirectors()
    {
    	return directors;
    }
}
