package org.ow2.opensuit.samples.movies_db.uibeans;

import org.ow2.opensuit.core.error.LocalizedError;
import org.ow2.opensuit.core.table.ITableRenderingContext;
import org.ow2.opensuit.samples.movies_db.model.dataaccess.DataBaseError;
import org.ow2.opensuit.samples.movies_db.model.dataaccess.MoviesDataBase;
import org.ow2.opensuit.samples.movies_db.model.dataaccess.RequestOptions;
import org.ow2.opensuit.samples.movies_db.model.to.MovieTO;



/**
 * This bean holds a list of movies.
 * 
 * It is initilized either by loading all movies from the database, or
 * by searching movies by title or director name.
 * 
 * <p>This version of the bean performs the search operation at each rendering,
 * including sort and pagination.
 */
public class ListOfMoviesBean4
{
	// ==========================================================
	// === Attributes
	// ==========================================================
	private String title;
	private String director;
	private int moviesCount = -1;
	
	// ==========================================================
	// === Control Methods
	// ==========================================================
	/**
	 * Loads all movies from the database
	 */
	public void loadAll() throws LocalizedError
	{
		try
        {
	        moviesCount = MoviesDataBase.getDataBase().countAllMovies();
        }
        catch(DataBaseError e)
        {
			throw new LocalizedError("database_technical", new Object[]{e.getMessage()});
        }
	}
	/**
	 * Searches movies by title
	 */
	public String searchByTitle() throws LocalizedError
	{
		try
        {
	        moviesCount = MoviesDataBase.getDataBase().countMovieByTitle(title);
        }
        catch(DataBaseError e)
        {
			throw new LocalizedError("database_technical", new Object[]{e.getMessage()});
        }
		if(moviesCount == 0)
			return "zero";
		if(moviesCount == 1)
			return "one";
		return "several";
	}
	/**
	 * Searches movies by director name
	 */
	public String searchByDirector() throws LocalizedError
	{
		try
        {
	        moviesCount = MoviesDataBase.getDataBase().countMoviesByDirector(director);
        }
        catch(DataBaseError e)
        {
			throw new LocalizedError("database_technical", new Object[]{e.getMessage()});
        }
		if(moviesCount == 0)
			return "zero";
		if(moviesCount == 1)
			return "one";
		return "several";
	}
	/**
	 * Resets the list
	 */
	public void reset()
	{
		moviesCount = -1;
		director = null;
		title = null;
	}
	/**
	 * Determines whether a list should be displayed
	 */
	public boolean hasList()
	{
		return moviesCount >= 0;
	}
	// ==========================================================
	// === Attributes Getters & Setters
	// ==========================================================
	public String getDirector()
    {
    	return director;
    }

	public void setDirector(String author)
    {
    	this.director = author;
    }

	public String getTitle()
    {
    	return title;
    }

	public void setTitle(String title)
    {
    	this.title = title;
    }

	public MovieTO[] getMovies() throws LocalizedError
    {
		// --- the search request is performed here, with sort and pagination options
		RequestOptions options = null;
		ITableRenderingContext ctx = ITableRenderingContext.get(null);
		if(ctx != null)
		{
			options = new RequestOptions();
			if(ctx.hasPagination())
			{
				options.setFirstResult(ctx.getFirstIndex());
				options.setNbOfResults(ctx.getRowsPerPage());
			}
			
			if(ctx.hasSortedColumn())
			{
				switch(ctx.getSortColumn())
				{
					case 0: //ID
						options.setSortCriteria(MoviesDataBase.SORT_BY_ID);
						break;
					case 1: // French Title
						options.setSortCriteria(MoviesDataBase.SORT_BY_FRENCH_TITLE);
						break;
					case 2: // Original Title
						options.setSortCriteria(MoviesDataBase.SORT_BY_ORIGINAL_TITLE);
						break;
					case 3: // Director
						options.setSortCriteria(MoviesDataBase.SORT_BY_DIRECTOR);
						break;
					case 4: // Release Date
						options.setSortCriteria(MoviesDataBase.SORT_BY_RELEASE_DATE);
						break;
				}
				options.setSortAscendingly(ctx.getSortDirection());
			}
		}
		
		try
		{
			if(title != null)
			{
				// --- search by title
				if(ctx != null)
					ctx.setRowsCount(MoviesDataBase.getDataBase().countMovieByTitle(title));
				return MoviesDataBase.getDataBase().searchMovieByTitle(title, options);
			}
			else if(director != null)
			{
				// --- search by director
				if(ctx != null)
					ctx.setRowsCount(MoviesDataBase.getDataBase().countMoviesByDirector(director));
				return MoviesDataBase.getDataBase().searchMoviesByDirector(director, options);
			}
			else
			{
				// --- view all movies
				if(ctx != null)
					ctx.setRowsCount(MoviesDataBase.getDataBase().countAllMovies());
				return MoviesDataBase.getDataBase().getAllMovies(options);
			}
		}
		catch(DataBaseError e)
		{
			throw new LocalizedError("database_technical", new Object[]{e.getMessage()});
		}
    }
}
