package org.ow2.opensuit.samples.movies_db.uibeans;

import java.util.ArrayList;
import java.util.List;

import org.ow2.opensuit.core.error.LocalizedError;
import org.ow2.opensuit.samples.movies_db.model.dataaccess.DataBaseError;
import org.ow2.opensuit.samples.movies_db.model.dataaccess.MoviesDataBase;
import org.ow2.opensuit.samples.movies_db.model.to.MovieTO;
import org.ow2.opensuit.samples.movies_db.model.to.PersonTO;



/**
 * This bean holds a list of movies.
 * 
 * It is initilized either by loading all movies from the database, or
 * by searching movies by title or director name.
 * 
 * <p>This version of the bean performs the search when loading, then
 * lets the framework manage sort and pagination.
 * This design may only be used if you're sure the amount of data is "reasonable"
 * and under control.
 */
public class ListOfMoviesBean5
{
	// ==========================================================
	// === Attributes
	// ==========================================================
	private String title;
	private String director;
	private MovieTO[] movies;
	private int[] selected;
	
	// ==========================================================
	// === Control Methods
	// ==========================================================
	/**
	 * Loads all movies from the database
	 */
	public void loadAll() throws LocalizedError
	{
		try
        {
	        movies = MoviesDataBase.getDataBase().getAllMovies(null);
        }
        catch(DataBaseError e)
        {
			throw new LocalizedError("database_technical", new Object[]{e.getMessage()});
        }
	}
	/**
	 * Searches movies by title
	 */
	public String searchByTitle() throws LocalizedError
	{
		try
        {
	        movies = MoviesDataBase.getDataBase().searchMovieByTitle(title, null);
        }
        catch(DataBaseError e)
        {
			throw new LocalizedError("database_technical", new Object[]{e.getMessage()});
        }
		if(movies == null || movies.length == 0)
			return "zero";
		if(movies.length == 1)
			return "one";
		return "several";
	}
	/**
	 * Searches movies by director name
	 */
	public String searchByDirector() throws LocalizedError
	{
		try
        {
	        movies = MoviesDataBase.getDataBase().searchMoviesByDirector(director, null);
        }
        catch(DataBaseError e)
        {
			throw new LocalizedError("database_technical", new Object[]{e.getMessage()});
        }
		if(movies == null || movies.length == 0)
			return "zero";
		if(movies.length == 1)
			return "one";
		return "several";
	}
	public String checkDelete()
	{
		if(selected == null || selected.length == 0)
		{
			System.out.println("No movie selected");
			return "no_movie";
		}
		System.out.println("Selected movies:");
		for(int i=0; i<selected.length; i++)
			System.out.println(" -"+i+": "+selected[i]);
		return "success";
	}
	public String delete() throws DataBaseError, LocalizedError
	{
		if(selected != null && selected.length > 0)
		{
			for(int i=0; i<selected.length; i++)
			{
//				System.out.println(" -"+i+": "+selected[i]);
				MoviesDataBase.getDataBase().removeMovie(selected[i]);
			}
			// --- reload and unselect
			loadAll();
			selected = null;
		}
		return "success";
	}
	/**
	 * Resets the list
	 */
	public void reset()
	{
		movies = null;
		director = null;
		title = null;
	}
	/**
	 * Determines whether a list should be displayed
	 */
	public boolean hasList()
	{
		return movies != null;
	}
	/**
	 * Returns a list of suggested director names
	 * @param name
	 * @return
	 */
	public List<String> suggestDirectorNames(String name)
	{
		// --- suggest for at least one letter
		if(name == null)
			return null;
		name = name.trim();
		if(name.length() == 0)
			return null;
		try
		{
			PersonTO[] directors = MoviesDataBase.getDataBase().searchDirectorByName(name, null);
			ArrayList<String> suggestions = new ArrayList<String>(directors.length);
			for(PersonTO d : directors)
				suggestions.add(d.getName());
			return suggestions;
		}
		catch (DataBaseError e)
		{
		}
		return null;
	}
	/**
	 * Returns a list of suggested movie titles
	 * @param name
	 * @return
	 */
	public List<String> suggestMovieTitles(String title)
	{
		// --- suggest for at least one letter
		if(title == null)
			return null;
		title = title.trim();
		if(title.length() == 0)
			return null;
		try
		{
			MovieTO[] movies = MoviesDataBase.getDataBase().searchMovieByTitle(title, null);
			ArrayList<String> suggestions = new ArrayList<String>(movies.length);
			for(MovieTO t : movies)
				suggestions.add(t.getOriginalTitle());
			return suggestions;
		}
		catch (DataBaseError e)
		{
		}
		return null;
	}
	// ==========================================================
	// === Attributes Getters & Setters
	// ==========================================================
	public String getDirector()
    {
    	return director;
    }

	public void setDirector(String author)
    {
    	this.director = author;
    }

	public String getTitle()
    {
    	return title;
    }

	public void setTitle(String title)
    {
    	this.title = title;
    }

	public MovieTO[] getMovies() throws LocalizedError
    {
		return movies;
    }
	public int[] getSelected()
    {
    	return selected;
    }
	public void setSelected(int[] selected)
    {
    	this.selected = selected;
    }

}
