/**
 * Open SUIT - Simple User Interface Toolkit
 * 
 * Copyright (c) 2009 France Telecom, http://www.francetelecom.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package org.ow2.opensuit.xml.spring;

import java.util.HashMap;

import javax.servlet.http.HttpServletRequest;

import org.ow2.opensuit.core.session.IPageContextListener;
import org.ow2.opensuit.core.session.OpenSuitSession;
import org.springframework.beans.factory.ObjectFactory;
import org.springframework.beans.factory.config.Scope;

/**
 * This class implements "page" scope for the Spring Framework.
 * This customised scope, specify to Spring that bean affected
 * with this scope are managed/hold just in the current page
 * context. When the page context is reinitialized the bean is
 * recreate.
 * In order to add this scope to you application, add the following
 * to your configuration:
 * <pre>
 * <bean class="org.springframework.beans.factory.config.CustomScopeConfigurer">
 *   <property name="scopes">
 *     <map>
 *       <entry key="page-context">
 *         <bean class="org.ow2.opensuit.core.components.spring.PageContext"/>
 *       </entry>
 *     </map>
 *   </property>
 * </bean>
 * </pre>
 * @author Adrien Ruffie / Pierre Smeyers
 * @author http://opensuit.ow2.org/
 * @version 1.0
 */
public class PageContext implements Scope
{

	/**
     * The main purpose of this method, is to retrieve the attribute linked
     * with the <code>name</code> provided in function parameter. If no attached attribute
     * was retrieved to the page context, an instance (possibly shared or
     * independent) of the object managed by this factory can be return.
     * @param name represent the name attribute which will be search into
     * page context. 
     * @param objectFactory Defines a factory which can return an Object
     * instance (possibly shared or independent) when invoked.
	 * @return the desired object (never <code>null</code>)
     * @see org.springframework.beans.factory.config.Scope#get(String, ObjectFactory)
     */
	public Object get(final String name, final ObjectFactory objectFactory)
	{
		final HttpServletRequest request = OpenSuitSession.getCurrentRequest();
		final OpenSuitSession session = OpenSuitSession.getSession(request);
		final org.ow2.opensuit.core.session.PageContext ctx = session.getCurrentPageContext();
		
		Object o = ctx.getAttribute(name);
		if(o == null)
		{
			o = objectFactory.getObject();
			
			if(o != null){
				ctx.setAttribute(name, o);
			}
		}
		return o;
	}

	/**
	 * @see org.springframework.beans.factory.config.Scope#getConversationId()
	 */
	public String getConversationId()
	{
		return null;
	}

	/**
	 * This method implements a destruction callback linked to,
	 * the page context scope. This callback can be execute in
	 * order to perform destruction of the specified object in
	 * the page scope
	 * @param name the name of the object located in the page
	 * context scope to execute the destruction callback
	 * @param callback the destruction callback to be executed.
	 * @see org.springframework.beans.factory.config.Scope#registerDestructionCallback(String, Runnable)
	 */
	public void registerDestructionCallback(final String name, final Runnable callback)
	{
		final HttpServletRequest request = OpenSuitSession.getCurrentRequest();
		final OpenSuitSession session = OpenSuitSession.getSession(request);
		final org.ow2.opensuit.core.session.PageContext ctx = session.getCurrentPageContext();
		
		HashMap<String, Runnable> name2Callback = (HashMap<String, Runnable>)ctx.getAttribute("_SPRING_DESTRUCT_CB");
		if(name2Callback == null)
		{
			ctx.addListener(new IPageContextListener(){
				public void onQuit(final HttpServletRequest request, final org.ow2.opensuit.core.session.PageContext pageContext)
				{
					final HashMap<String, Runnable> name2Callback = (HashMap<String, Runnable>)pageContext.getAttribute("_SPRING_DESTRUCT_CB");
					for(final Runnable cb : name2Callback.values())
					{
						cb.run();
					}
				}
			});
			name2Callback = new HashMap<String, Runnable>(1);
			ctx.setAttribute("_SPRING_DESTRUCT_CB", name2Callback);
		}
		name2Callback.put(name, callback);
	}

	/**
	 * Remove the object with the given <code>name</code> from the 
	 * page context scope.
	 * <p>Returns <code>null</code> if no object was found; otherwise
	 * returns the removed <code>Object</code> from page context.
	 * @param name the name of the object to remove from page context
	 * @return the removed object, or <code>null</code> if object
	 * wasn't present in page context
	 * @see org.springframework.beans.factory.config.Scope#remove(String)
	 */
	public Object remove(final String name)
	{
		final HttpServletRequest request = OpenSuitSession.getCurrentRequest();
		final OpenSuitSession session = OpenSuitSession.getSession(request);
		final org.ow2.opensuit.core.session.PageContext ctx = session.getCurrentPageContext();
		
		final Object o = ctx.getAttribute(name);
		
		if(o != null){
			ctx.removeAttribute(name);
		}
		
		return o;
	}
	
}
