/**
 * Open SUIT - Simple User Interface Toolkit
 * 
 * Copyright (c) 2009 France Telecom, http://www.francetelecom.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.ow2.opensuit.xml.spring;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Type;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;

import javax.servlet.ServletContext;
import javax.servlet.http.HttpServletRequest;

import org.ow2.opensuit.xml.base.binding.IDataSource;
import org.ow2.opensuit.xmlmap.annotations.XmlAttribute;
import org.ow2.opensuit.xmlmap.annotations.XmlChildren;
import org.ow2.opensuit.xmlmap.annotations.XmlDoc;
import org.ow2.opensuit.xmlmap.annotations.XmlElement;
import org.ow2.opensuit.xmlmap.interfaces.IInitializable;
import org.ow2.opensuit.xmlmap.interfaces.IInitializationSupport;
import org.ow2.opensuit.xmlmap.interfaces.IInstantiationContext;

/**
 * This class allows to use :
 * <ul>
 * <li>Spring based beans
 * <li>Run-time validation for declared Spring beans
 * <li>Build-time validation for declared Spring beans
 * </ul>
 * 
 * @author Adrien Ruffie / Pierre Smeyers
 * @author http://opensuit.ow2.org/
 * @version 1.0
 */
@XmlDoc("Allow to use several beans, declared into Spring context.")
@XmlElement
public class SpringBeans implements IDataSource, IInitializable {

	@XmlDoc("Spring configuration file(s).<br/>" +
			"Note: This is not required at runtime if 'CreateContext' is false, but still it " +
			"is useful to enable build-time validation.")
	@XmlChildren(name = "ConfigFiles",minOccurs=1)
	private XmlConfig[] configFiles;

    @XmlDoc("Defines where the Spring configuration file(s) should be searched.<br/>" +
    		"<ul>" + 
    		"<li>WAR: in the WAR /WEB-INF directory." + 
            "<li>CLASSPATH: in the application classpath." +
            "</ul>" +
            "Default: WAR.")
    @XmlAttribute(name = "SearchIn", required = false)
    private ConfigLocation searchIn = ConfigLocation.WAR;
    
	@XmlDoc("Determines whether Open SUIT has to create and initialize the Spring Application Context.<br/>" +
			"If set to 'true', your Spring Application Context will be instantiated by Open SUIT, and made available " +
			"for the whole application. As a result you won't have to declare it anywhere else.<br/>" +
			"Otherwise Open SUIT assumes the Spring Application Context is declared and initialized by some other way (Spring servlet for " +
			"instance), and will just retrieve it through the J2EE servlet context.<br/>" +
			"Default: false.")
	@XmlAttribute(name = "CreateContext", required = false)
	private boolean createContext = false;

	/*
	 * This attribute can refer to, all type of Spring application context, but
	 * WebApplicationContext is really expected
	 */
	private Object appContext;

	/*
	 * Serve to retrieve the getType() method linked to the Spring application
	 * context used (in order to support all Spring version)
	 */
	private Method getType;

	/*
	 * Serve to retrieve the getBean() method linked to the Spring application
	 * context used (in order to support all Spring version)
	 */
	private Method getBean;

	/**
	 * Initialization method called after instantiation, in order to perform
	 * initialize attributes and validation.
	 * 
	 * @param initSupport
	 *            IInitializationSupport interface allows to log validation
	 *            errors, and trigger dependent objects initialization
	 * @param instContext
	 *            IInstantiationContext interface allows to instantiate an XML
	 *            document with external context (providing external ancestors)
	 * @see org.ow2.opensuit.xmlmap.interfaces.IInitializable#initialize(IInitializationSupport,
	 *      IInstantiationContext)
	 */
	public void initialize(final IInitializationSupport initSupport, final IInstantiationContext instContext) {
		
		final ServletContext servletCtx = (ServletContext) instContext.getAncestor(ServletContext.class);
		boolean isBuildTimeValidation = servletCtx.getClass().getName().startsWith("org.ow2.opensuit.core.impl.tools");
		
		// --- 1: initialize Spring context and retrieve classes and methods
		if (createContext || isBuildTimeValidation)
		{
			// ---  we have to setup a Spring context
			if (configFiles == null)
				// --- error already added (by XMLMap)
				return;

			// Paths of all application context files from war
			final List<String> files = new ArrayList<String>();

			// For each application context file, try to retrieve its URL
			for (final XmlConfig f : configFiles) {
				files.add(f.getFile());
			}

			/*
			 * Now try to instantiate the Spring Application Context (either War or Classpath)
			 */
			if(searchIn == ConfigLocation.CLASSPATH)
				appContext = newClasspathAppContext(files, initSupport);
			else
				appContext = newWarAppContext(files, initSupport, servletCtx);
			
			if(appContext == null)
				// --- error already added
				return;

			// --- retrieve getBean() and getType() methods
			try {
				this.getBean = appContext.getClass().getMethod("getBean", String.class);
				this.getType = appContext.getClass().getMethod("getType", String.class);
			} catch (final SecurityException e) {
				initSupport.addValidationMessage(this, null,
						IInitializationSupport.ERROR,
						"Security exception throwed by the security manager. " + e.getMessage());
			} catch (final NoSuchMethodException e) {
				initSupport
						.addValidationMessage(this, null,
								IInitializationSupport.ERROR,
								"Ooops! Looks like you're using a Spring version unsupported by Open SUIT.");
			}
		}
		else
		{
			// --- assume the Spring context is already set and accessible in the servlet context
			try {
				final Class<?> wacuClass = Class.forName("org.springframework.web.context.support.WebApplicationContextUtils");
				final Method gwactMethod = wacuClass.getMethod("getWebApplicationContext", ServletContext.class);
				appContext = gwactMethod.invoke(wacuClass, servletCtx);
				if (appContext == null)
				{
					initSupport.addValidationMessage(this, null, IInitializationSupport.ERROR,
							"No Spring context found in servlet context. \nMake sure it is properly defined, and that it is setup before starting the Open SUIT servlet!");
					return;
				}

				this.getType = appContext.getClass().getMethod("getType", String.class);
				this.getBean = appContext.getClass().getMethod("getBean", String.class);

			} catch (final ClassNotFoundException e) {
				// --- Spring classes not found: classpath problem
				initSupport.addValidationMessage(this, null, IInitializationSupport.ERROR,
								"Could not load spring context. Spring library is probably not in your classpath.");
			} catch (final NoSuchMethodException e) {
				initSupport.addValidationMessage(this, null,
								IInitializationSupport.ERROR,
								"Ooops! Looks like you're using a Spring version unsupported by Open SUIT.");
			} catch (final InvocationTargetException e) {
				// --- error while invoking
				// WebApplicationContextUtils.getWebApplicationContext():
				// Spring
				// init problem
				initSupport.addValidationMessage(this, null, IInitializationSupport.ERROR,
						"Error in Spring initialization. Please look at Spring logs. "
								+ e.getCause());
			} catch (final Exception e) {
				// --- error while invoking
				// WebApplicationContextUtils.getWebApplicationContext():
				// Spring
				// init problem
				initSupport.addValidationMessage(this, null,
						IInitializationSupport.ERROR,
						"Error while retrieving Spring context: "
								+ e.getCause());
			}
		}
	}

	/**
	 * Research method called in order to find the type (wrapped in form of
	 * <code>Class<?></code>, of the asked bean <code>name</code> provided
	 * in function parameter. Can return <code>null</code> if the bean isn't
	 * exist.
	 * 
	 * @param name
	 *            Represent the called bean <code>name</code> in Spring
	 *            Context
	 * @see org.ow2.opensuit.xml.interfaces.IBeanProvider#getBeanType(String)
	 * @return Class<?> return the type (wrapped in <code>Class<?></code>
	 *         object) of asked bean
	 * @throws UnresolvedBeanError
	 *             this means that the bean resolution failed
	 */
	public Class<?> getBeanType(final String name) throws UnresolvedBeanError {
		if (this.getType == null) {
			return null;
		}
		try {
			final Class<?> clazz = (Class<?>) this.getType.invoke(appContext, name);
			return clazz;
		} catch (final InvocationTargetException e) {
			// --- probably a NoSuchBeanDefinition
			throw new UnresolvedBeanError();
		} catch (final Exception e) {
			e.printStackTrace();
			return null;
		}
	}

	/**
	 * Research method called in order to find the <code>Type</code>, of the
	 * asked bean <code>name</code> provided in function parameter. Can return
	 * <code>null</code> if the bean isn't exist. Unlike getBeanType method,
	 * getBeanGenericType return the bean <code>Type</code> unwrapped (not
	 * return <code>Class<?></code> but <code>Type</code>)
	 * 
	 * @param name
	 *            Represent the called bean <code>name</code> in Spring
	 *            Context
	 * @see org.ow2.opensuit.xml.interfaces.IBeanProvider#getBeanGenericType(String)
	 * @return Type return the <code>Type</code> of asked bean
	 * @throws UnresolvedBeanError
	 *             this means that the bean resolution failed
	 */
	public Type getBeanGenericType(final String name)
			throws UnresolvedBeanError {
		return getBeanType(name);
	}

	/**
	 * Method called in order to retrieve the associated value for the asked
	 * bean <code>name</code> provided in function parameter. Can return
	 * <code>null</code> if the bean isn't exist.
	 * 
	 * @param request
	 *            Unused parameter
	 * @param name
	 *            Represent the called bean <code>name</code> in Spring
	 *            Context
	 * @see org.ow2.opensuit.xml.interfaces.IBeanProvider#getBeanValue(HttpServletRequest,
	 *      String)
	 * @return Object return the value of asked bean wrapped in Object instance,
	 *         because bean value can be of any type like <code>Integer</code>,
	 *         <code>String</code>, <code>DataSource</code> ...
	 * @throws Exception
	 *             Can throws several types of exception, but one specifically
	 *             can be throw 'InvocationTargetException'.
	 */
	public Object getBeanValue(final HttpServletRequest request,
			final String name) throws Exception {

		if (this.getBean == null) {
			return null;
		}
		try {
			Object obj = this.getBean.invoke(appContext, name);
			return obj;
		} catch (final InvocationTargetException e) {
			if (e.getCause() instanceof Exception)
				throw (Exception) e.getCause();
			if (e.getCause() instanceof Error)
				throw (Error) e.getCause();
			e.printStackTrace(System.err);
			return null;
		}
	}

	/**
	 * Allow to instantiate an FileSystem XML Spring Application Context by
	 * aggregating all provided app context files.
	 * 
	 * @param warFiles
	 *            all paths of app context files located in war
	 * @param initSupport
	 *            allow to log several exception
	 * @param servletContext
	 *            This Servlet Context allow to retrieve app context from, its
	 *            context.
	 * @return <code>Object</code> return the application context which
	 *         aggregate all app context files provided in war
	 */
	private Object newWarAppContext(final List<String> warFiles,
			final IInitializationSupport initSupport,
			final ServletContext servletContext) {

		final List<String> urlContextList = new ArrayList<String>();

		for (final String warFile : warFiles) {

			try {
				final URL url = servletContext.getResource("/WEB-INF" + warFile);

				if (url == null) {
					initSupport.addValidationMessage(this, null,
							IInitializationSupport.ERROR, "Bad searchIn for: "
									+ warFile + " application context file, "
									+ "declared in SpringBeans tag. It must "
									+ "be located in /WEB-INF");
					return null;
				}
				urlContextList.add(url.toString());
			} catch (final MalformedURLException e) {
				initSupport.addValidationMessage(this, null,
						IInitializationSupport.ERROR, "Malformed URL for: "
								+ warFile + " application context file, "
								+ "declared in SpringBeans tag");
			}
		}
		final String[] s = urlContextList.toArray(new String[urlContextList.size()]);
		try {
			final Class<?> fsxacClass = Class.forName("org.springframework.context.support.FileSystemXmlApplicationContext");
			final Constructor<?> fsxacConst = fsxacClass.getConstructor(String[].class);
			return fsxacConst.newInstance(new Object[] { s });
		} catch (final ClassNotFoundException e) {
			// --- Spring classes not found: classpath problem
			initSupport
					.addValidationMessage(
							this,
							null,
							IInitializationSupport.ERROR,
							"Could not load Spring context. Spring library is probably not in your classpath.");
		} catch (final NoSuchMethodException e) {
			initSupport
					.addValidationMessage(this, null,
							IInitializationSupport.ERROR,
							"Ooops! Looks like you're using a Spring version unsupported by Open SUIT.");
		} catch (final InvocationTargetException e) {
			// --- error while invoking
			// WebApplicationContextUtils
			// .getWebApplicationContext(): Spring init problem
			initSupport.addValidationMessage(this, null,
					IInitializationSupport.ERROR,
					"Error in Spring initialization. Please look at Spring logs. "
							+ e.getCause());
		} catch (final Exception e) {
			// --- error while invoking
			// WebApplicationContextUtils
			// .getWebApplicationContext(): Spring init problem
			initSupport.addValidationMessage(this, null,
					IInitializationSupport.ERROR,
					"Error while retrieving Spring context: " + e.getCause());
		}
		return null;
	}

	/**
	 * Allow to instantiate an ClassPath XML Spring Application Context by
	 * aggregating all provided app context files from classpath.
	 * 
	 * @param classpathURLs
	 *            all URLs of app context files located in classpath
	 * @param initSupport
	 *            allow to log several exception
	 * @param warAgregatedApplicationContext
	 *            this application context can be aggregated (like parent
	 *            application context), with the app context which will be
	 *            instantiated thank to classpath in this function .
	 * @return <code>Object</code> return the application context which
	 *         aggregate all classpath app context files provided, and parent
	 *         war app context if is provided.
	 */
	private Object newClasspathAppContext(
			final List<String> classpathURLs,
			final IInitializationSupport initSupport) {

		final String[] s = classpathURLs.toArray(new String[classpathURLs.size()]);
		// TODO: check every file url with getClassLoader().getResource()...
		try {
			final Class<?> fsxacClas = Class.forName("org.springframework.context.support.ClassPathXmlApplicationContext");
//			Constructor<?> fsxacConst = null;
			/*
			 * If parent context is provided invoke constructor with parent
			 * context, else without parent context.
			 */
//			if (warAgregatedApplicationContext != null) {
//
//				Class appContextClass = Class.forName("org.springframework.context.ApplicationContext");
//				fsxacConst = fsxacClas.getConstructor(String[].class, appContextClass);
//				appContext = fsxacConst.newInstance(new Object[] { s, warAgregatedApplicationContext });
//			} else {

			Constructor<?> fsxacConst = fsxacClas.getConstructor(String[].class);
			return fsxacConst.newInstance(new Object[] { s });
//			}
		} catch (final ClassNotFoundException e) {
			// --- Spring classes not found: classpath problem
			initSupport
					.addValidationMessage(
							this,
							null,
							IInitializationSupport.ERROR,
							"Could not load Spring context. Spring library is probably not in your classpath.");
		} catch (final NoSuchMethodException e) {
			initSupport
					.addValidationMessage(this, null,
							IInitializationSupport.ERROR,
							"Ooops! Looks like you're using a Spring version unsupported by Open SUIT.");
		} catch (final InvocationTargetException e) {
			// --- error while invoking
			// WebApplicationContextUtils.getWebApplicationContext():
			// Spring init problem
			initSupport.addValidationMessage(this, null,
					IInitializationSupport.ERROR,
					"Error in Spring initialization. Please look at Spring logs. "
							+ e.getCause());
		} catch (final Exception e) {
			// --- error while invoking
			// WebApplicationContextUtils.getWebApplicationContext():
			// Spring init problem
			initSupport.addValidationMessage(this, null,
					IInitializationSupport.ERROR,
					"Error while retrieving Spring context: " + e.getCause());
		}
		return null;
	}
}
