/*
 * Copyright (C) 2011  Bull S. A. S.
 * Bull, Rue Jean Jaures, B.P.68, 78340, Les Clayes-sous-Bois
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation
 * version 2.1 of the License.
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA  02110-1301, USA.
 */

package org.ow2.orchestra.common.gwt.soapui.client.presenters.wsdl.explorer;

import com.google.gwt.core.client.GWT;
import com.google.gwt.event.logical.shared.HasOpenHandlers;
import com.google.gwt.event.logical.shared.OpenEvent;
import com.google.gwt.event.logical.shared.SelectionEvent;
import com.google.gwt.event.shared.EventBus;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.google.gwt.user.client.ui.TreeItem;
import java.util.List;
import org.ow2.orchestra.common.gwt.soapui.client.SoapUIServices;
import org.ow2.orchestra.common.gwt.soapui.client.SoapUIServicesAsync;
import org.ow2.orchestra.common.gwt.soapui.client.events.ExploreWsdlEvent;
import org.ow2.orchestra.common.gwt.soapui.client.events.OpenOperationEvent;
import org.ow2.orchestra.common.gwt.soapui.client.model.WsdlInterface;
import org.ow2.orchestra.common.gwt.soapui.client.model.WsdlOperation;
import org.ow2.orchestra.common.gwt.soapui.client.model.WsdlProject;
import org.ow2.orchestra.common.gwt.soapui.client.ui.wsdl.explorer.WsdlExplorer;

/**
 * @author Loic Albertin
 */
public class WsdlPresenterImpl implements WsdlExplorer.WsdlPresenter {

  private static final String OPERATION_PREFIX = "operation: ";
  private static final String INTERFACE_PREFIX = "interface: ";
  private WsdlExplorer view;

  private EventBus eventBus;

  public WsdlPresenterImpl(WsdlExplorer view, EventBus bus) {
    this.view = view;
    this.eventBus = bus;

    bus.addHandler(ExploreWsdlEvent.TYPE, new ExploreWsdlEvent.ExploreWsdlEventHandler() {
      public void onExploreWsdl(ExploreWsdlEvent event) {
        WsdlPresenterImpl.this.exploreWsdl(event.getWsdlUrl());
      }
    });
  }

  private SoapUIServicesAsync soapUIServices = GWT.create(SoapUIServices.class);


  public void onOpen(final OpenEvent<TreeItem> treeItemOpenEvent) {
    final TreeItem targetItem = treeItemOpenEvent.getTarget();
    if (targetItem.getUserObject() != null && targetItem.getUserObject() instanceof WsdlInterface) {
      if (targetItem.getChildCount() == 1 && "".equals(targetItem.getChild(0).getText())) {
        // Close the item immediately
        targetItem.setState(false, false);
        // Remove the temporary item when we finish loading
        targetItem.getChild(0).remove();
        final WsdlInterface wsdlInterface = (WsdlInterface) targetItem.getUserObject();
        soapUIServices.getOperations(wsdlInterface.getWsdlProject().getWsdlUrl(),
            wsdlInterface.getInterfaceName(),
            new AsyncCallback<List<String>>() {
              public void onFailure(Throwable caught) {
                GWT.log("Unable to access soapui!\n" + caught.getMessage());
              }

              public void onSuccess(List<String> result) {
                for (String s : result) {
                  WsdlOperation wsdlOperation = wsdlInterface.addOperation(s);
                  TreeItem item = targetItem.addItem(OPERATION_PREFIX + s);
                  item.setUserObject(wsdlOperation);
                }
                // Reopen the item
                targetItem.setState(true, false);
              }
            });

      } else {
        targetItem.setState(true, false);
      }
    }
  }

  public void clear() {
    this.view.clear();
  }

  public void exploreWsdl(final String wsdlUrl) {
    final WsdlProject model = new WsdlProject(wsdlUrl);
    soapUIServices.getInterfaces(wsdlUrl, new AsyncCallback<List<String>>() {
      public void onFailure(Throwable caught) {
        GWT.log("Unable to access soapui!\n" + caught.getMessage());
      }

      public void onSuccess(List<String> result) {
        for (String s : result) {
          WsdlInterface wsdlInterface = model.addInterfaceModel(s);
          TreeItem item = WsdlPresenterImpl.this.view.addItem(INTERFACE_PREFIX + s);
          item.setUserObject(wsdlInterface);
          //Temporary add an empty item in order to make item expendable
          item.addItem("");
        }
      }
    });
  }

  public void exploreModel(WsdlProject model) {
    if (model.getWsdlInterfaces().size() == 0) {
      exploreWsdl(model.getWsdlUrl());
    } else {
      for (WsdlInterface wsdlInterface : model.getWsdlInterfaces()) {
        TreeItem interfaceTreeItem = this.view.addItem(INTERFACE_PREFIX + wsdlInterface.getInterfaceName());
        interfaceTreeItem.setUserObject(wsdlInterface);
        if (wsdlInterface.getOperations().size() > 0) {
          for (WsdlOperation wsdlOperation : wsdlInterface.getOperations()) {
            TreeItem operationTreeItem = interfaceTreeItem.addItem(OPERATION_PREFIX +
                wsdlOperation.getOperationName());
            operationTreeItem.setUserObject(wsdlOperation);
          }
        } else {
          //Temporary add an empty item in order to make item expendable
          interfaceTreeItem.addItem("");
        }
      }
    }
  }

  public void onSelection(SelectionEvent<TreeItem> treeItemSelectionEvent) {
    final TreeItem targetItem = treeItemSelectionEvent.getSelectedItem();
    if (targetItem.getUserObject() instanceof WsdlOperation){
      WsdlOperation wsdlOperation = (WsdlOperation) targetItem.getUserObject();
      eventBus.fireEvent(new OpenOperationEvent(wsdlOperation));
    } else if (targetItem.getUserObject() instanceof WsdlInterface) {
      OpenEvent.fire((HasOpenHandlers<TreeItem>) treeItemSelectionEvent.getSource(), targetItem);
    }
  }
}
