/*
 * Copyright (C) 2011 Bull S. A. S.
 * Bull, Rue Jean Jaures, B.P.68, 78340, Les Clayes-sous-Bois
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation
 * version 2.1 of the License.
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA  02110-1301, USA.
 */

package org.ow2.orchestra.common.gwt.soapui.client.ui.ws.interactions.impl;

import com.google.gwt.core.client.GWT;
import com.google.gwt.core.client.Scheduler;
import com.google.gwt.dom.client.Style;
import com.google.gwt.event.dom.client.MouseDownEvent;
import com.google.gwt.event.dom.client.MouseDownHandler;
import com.google.gwt.event.dom.client.MouseUpEvent;
import com.google.gwt.event.dom.client.MouseUpHandler;
import com.google.gwt.event.logical.shared.ResizeEvent;
import com.google.gwt.event.logical.shared.ResizeHandler;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.resources.client.ClientBundle;
import com.google.gwt.resources.client.ImageResource;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.ui.FlowPanel;
import com.google.gwt.user.client.ui.Image;
import com.google.gwt.user.client.ui.LayoutPanel;
import com.google.gwt.user.client.ui.TabLayoutPanel;
import com.google.gwt.user.client.ui.Widget;
import org.ow2.orchestra.common.gwt.soapui.client.events.MouseStillDownEvent;
import org.ow2.orchestra.common.gwt.soapui.client.events.MouseStillDownTimer;

/**
 * @author Loic Albertin
 *         based on https://groups.google.com/group/google-web-toolkit/browse_thread/thread/c0df252d4db7c0f0/af9a3df602ac00f9?hl=de&lnk=gst&q=tablayoutpanel&pli=1#af9a3df602ac00f9
 *         A {@link TabLayoutPanel} that shows scroll buttons if necessary
 */
public class ScrollableTabLayoutPanel extends TabLayoutPanel {

  public interface Resources extends ClientBundle {
    @Source("leftArrowImage.png")
    ImageResource leftArrowImage();

    @Source("rightArrowImage.png")
    ImageResource rightArrowImage();
  }

  private static final Resources RESOURCES = GWT.create(Resources.class);

  private static final int IMAGE_PADDING_PIXELS = 4;
  private LayoutPanel panel;
  private FlowPanel tabBar;
  private Image scrollLeftButton;
  private Image scrollRightButton;
  private HandlerRegistration windowResizeHandler;


  private ImageResource leftArrowImage;
  private ImageResource rightArrowImage;

  private MouseStillDownTimer mouseStillDownTimer = new MouseStillDownTimer(200);

  public ScrollableTabLayoutPanel(double barHeight, Style.Unit barUnit,
                                  ImageResource leftArrowImage, ImageResource rightArrowImage) {
    super(barHeight, barUnit);
    // The main widget wrapped by this composite, which is a LayoutPanel
    // with the tab bar&the tab content
    panel = (LayoutPanel) getWidget();
    // Find the tab bar, which is the first flow panel in the LayoutPanel
    for (int i = 0; i < panel.getWidgetCount(); ++i) {
      Widget widget = panel.getWidget(i);
      if (widget instanceof FlowPanel) {
        tabBar = (FlowPanel) widget;
        break; // tab bar found
      }
    }

    this.leftArrowImage = leftArrowImage;
    this.rightArrowImage = rightArrowImage;
    initScrollButtons();
  }

  @Override
  public void add(Widget child, Widget tab) {
    super.add(child, tab);
    checkIfScrollButtonsNecessary();
  }

  @Override
  public boolean remove(Widget w) {
    boolean b = super.remove(w);
    checkIfScrollButtonsNecessary();
    return b;
  }

  @Override
  protected void onLoad() {
    super.onLoad();
    if (windowResizeHandler == null) {
      windowResizeHandler = Window.addResizeHandler(new ResizeHandler() {
        public void onResize(ResizeEvent event) {
          checkIfScrollButtonsNecessary();
        }
      });
    }
  }

  @Override
  protected void onUnload() {
    super.onUnload();
    if (windowResizeHandler != null) {
      windowResizeHandler.removeHandler();
      windowResizeHandler = null;
    }
  }

  public void handleScrollRequest(final int diff) {
    Widget lastTab = getLastTab();
    if (lastTab == null)
      return;
    int newLeft = parsePosition(tabBar.getElement().getStyle().getLeft()) + diff;
    int rightOfLastTab = getRightOfWidget(lastTab);
    // Prevent scrolling the last tab too far away form the right border,
    // or the first tab further than the left border position
    if (newLeft > 0) {
      newLeft = 0;
    } else if ((getTabBarWidth() - newLeft > (rightOfLastTab + 20))) {
        newLeft = -(rightOfLastTab - getTabBarWidth());
    }
    scrollTo(newLeft);
  }

  /**
   * Create and attach the scroll button images with a click handler
   */
  private void initScrollButtons() {
    scrollLeftButton = new Image(leftArrowImage);
    int leftImageWidth = scrollLeftButton.getWidth();
    panel.insert(scrollLeftButton, 0);

    panel.setWidgetLeftWidth(scrollLeftButton, 0, Style.Unit.PX,
        leftImageWidth, Style.Unit.PX);
    panel.setWidgetTopHeight(scrollLeftButton, getTabBarHeight() / 2 + scrollLeftButton.getHeight() / 2, Style.Unit.PX,
        scrollLeftButton.getWidth(), Style.Unit.PX);

    scrollLeftButton.addMouseDownHandler(new MouseDownHandler() {
      public void onMouseDown(MouseDownEvent event) {
        ScrollableTabLayoutPanel.this.handleScrollRequest(40);
        ScrollableTabLayoutPanel.this.mouseStillDownTimer.registerHandler(new MouseStillDownEvent.OnMouseStillDownEventHandler() {
          public void onMouseStillDown(MouseStillDownEvent event) {
            ScrollableTabLayoutPanel.this.handleScrollRequest(40);
          }
        });
      }
    });
    scrollLeftButton.addMouseUpHandler(new MouseUpHandler() {
      public void onMouseUp(MouseUpEvent event) {
        ScrollableTabLayoutPanel.this.mouseStillDownTimer.stopRecurring();
      }
    });
    scrollLeftButton.setVisible(false);
    scrollRightButton = new Image(rightArrowImage);
    panel.insert(scrollRightButton, 0);

    panel.setWidgetRightWidth(scrollRightButton, 0, Style.Unit.PX, scrollRightButton.getWidth(), Style.Unit.PX);
    panel.setWidgetTopHeight(scrollRightButton, getTabBarHeight() / 2 + scrollRightButton.getHeight() / 2, Style.Unit.PX,
        scrollRightButton.getHeight(), Style.Unit.PX);

    scrollRightButton.addMouseDownHandler(new MouseDownHandler() {
      public void onMouseDown(MouseDownEvent event) {
        ScrollableTabLayoutPanel.this.handleScrollRequest(-40);
        ScrollableTabLayoutPanel.this.mouseStillDownTimer.registerHandler(new MouseStillDownEvent.OnMouseStillDownEventHandler() {
          public void onMouseStillDown(MouseStillDownEvent event) {
            ScrollableTabLayoutPanel.this.handleScrollRequest(-40);
          }
        });
      }
    });
    scrollRightButton.addMouseUpHandler(new MouseUpHandler() {
      public void onMouseUp(MouseUpEvent event) {
        ScrollableTabLayoutPanel.this.mouseStillDownTimer.stopRecurring();
      }
    });
    scrollRightButton.setVisible(false);
  }

  private void checkIfScrollButtonsNecessary() {
    // Defer size calculations until sizes are available, when
    // calculating immediately after
    // add(), all size methods return zero
    Scheduler.get().scheduleDeferred(new Scheduler.ScheduledCommand() {
      public void execute() {
        boolean isScrolling = isScrollingNecessary();
        // When the scroll buttons are being hidden, reset the scroll
        // position to zero to
        // make sure no tabs are still out of sight
        if (scrollRightButton.isVisible() && !isScrolling) {
          resetScrollPosition();
        }
        scrollRightButton.setVisible(isScrolling);
        scrollLeftButton.setVisible(isScrolling);
        if (isScrolling) {
          panel.setWidgetLeftRight(tabBar, scrollLeftButton.getWidth(), Style.Unit.PX,
              scrollRightButton.getWidth(), Style.Unit.PX);
        } else {
          panel.setWidgetLeftRight(tabBar, 0, Style.Unit.PX,
              0, Style.Unit.PX);
        }
      }
    });

  }

  private void resetScrollPosition() {
    scrollTo(0);
  }

  private void scrollTo(int pos) {
    tabBar.getElement().getStyle().setLeft(pos, Style.Unit.PX);
  }

  private boolean isScrollingNecessary() {
    Widget lastTab = getLastTab();
    return lastTab != null && getRightOfWidget(lastTab) > getTabBarWidth();
  }

  private int getRightOfWidget(Widget widget) {
    return widget.getElement().getOffsetLeft() +
        widget.getElement().getOffsetWidth();
  }

  private int getTabBarWidth() {
    return tabBar.getElement().getParentElement().getClientWidth();
  }

  private int getTabBarHeight() {
    return tabBar.getElement().getParentElement().getClientHeight();
  }

  private Widget getLastTab() {
    if (tabBar.getWidgetCount() == 0)
      return null;
    return tabBar.getWidget(tabBar.getWidgetCount() - 1);
  }

  private static int parsePosition(String positionString) {
    int position;
    try {
      for (int i = 0; i < positionString.length(); i++) {
        char c = positionString.charAt(i);
        if (c != '-' && !(c >= '0' && c <= '9')) {
          positionString = positionString.substring(0, i);
        }
      }
      position = Integer.parseInt(positionString);
    } catch (NumberFormatException ex) {
      position = 0;
    }
    return position;
  }

  private boolean isLeftOfWidgetVisible(final Widget widget) {
    int position = parsePosition(tabBar.getElement().getStyle().getLeft());
    int leftOfWidget = widget.getElement().getOffsetLeft();
    int firstVisiblePosition = -position;
    return leftOfWidget > firstVisiblePosition;
  }

  private boolean isRightOfWidgetVisible(final Widget widget) {
    int tabBarWidth = getTabBarWidth();
    int position = parsePosition(tabBar.getElement().getStyle().getLeft());
    int rightOfWidget = getRightOfWidget(widget);
    int lastVisiblePosition = tabBarWidth - position;
    return rightOfWidget < lastVisiblePosition;
  }

  @Override
  public void selectTab(final int index, boolean fireEvents) {
    super.selectTab(index, fireEvents);
    Scheduler.get().scheduleDeferred(new Scheduler.ScheduledCommand() {
      public void execute() {
        Widget widget = tabBar.getWidget(index);
        int rightOfWidget = getRightOfWidget(widget);
        int tabBarWidth = getTabBarWidth();
        int position = parsePosition(tabBar.getElement().getStyle().getLeft());
        if (!isRightOfWidgetVisible(widget)) {
          handleScrollRequest(tabBarWidth - position - rightOfWidget);
        }
        if (!isLeftOfWidgetVisible(widget)) {
          scrollTo(-widget.getElement().getOffsetLeft());
        }
      }
    });


  }
}

