/*
 * Copyright (C) 2011  Bull S. A. S.
 * Bull, Rue Jean Jaures, B.P.68, 78340, Les Clayes-sous-Bois
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation
 * version 2.1 of the License.
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA  02110-1301, USA.
 */

package org.ow2.orchestra.common.gwt.soapui.server;

import com.eviware.soapui.DefaultSoapUICore;
import com.eviware.soapui.SoapUI;
import com.eviware.soapui.SoapUICore;
import com.eviware.soapui.impl.WsdlInterfaceFactory;
import com.eviware.soapui.impl.wsdl.WsdlInterface;
import com.eviware.soapui.impl.wsdl.WsdlOperation;
import com.eviware.soapui.impl.wsdl.WsdlProject;
import com.eviware.soapui.impl.wsdl.WsdlRequest;
import com.eviware.soapui.impl.wsdl.WsdlSubmit;
import com.eviware.soapui.impl.wsdl.WsdlSubmitContext;
import com.eviware.soapui.model.iface.Operation;
import com.eviware.soapui.model.iface.Request;
import com.eviware.soapui.model.iface.Response;
import com.eviware.soapui.settings.ProxySettings;
import com.eviware.soapui.support.SoapUIException;
import com.google.gwt.user.server.rpc.RemoteServiceServlet;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Pattern;
import javax.servlet.ServletException;
import org.apache.xmlbeans.XmlException;
import org.apache.xmlbeans.XmlObject;
import org.ow2.orchestra.common.gwt.soapui.client.ServiceException;
import org.ow2.orchestra.common.gwt.soapui.client.SoapUIServices;
import org.ow2.orchestra.util.Misc;

/**
 * @author Loic Albertin
 */
public class SoapUIServlet extends RemoteServiceServlet implements SoapUIServices {
  private static final long serialVersionUID = -80411514041641896L;

  private static final String TMP_DIR = System.getProperty("java.io.tmpdir", ".");
  /**
   * logger used to log out the exceptions
   */
  protected static final Logger LOG = Logger.getLogger(SoapUIServlet.class.getName());

  /**
   * Quoting special char as SoapUI will use regular expressions
   * According to http://docs.oracle.com/javase/6/docs/technotes/guides/net/proxies.html nonProxyHosts may
   * starts and/or ends with a '*' character while wildcard for RE is ".*"
   *
   * @param nonProxyHosts non proxy hosts in system property format
   * @return non proxy hosts in SoapUI expected format
   */
  public static String generateNonProxyHostsInSoapUIFormat(final String nonProxyHosts) {
    String[] patterns = nonProxyHosts.split("\\|");
    StringBuilder sb = new StringBuilder();
    for (String pattern : patterns) {
      pattern = pattern.replace("\"", "");
      pattern = pattern.replace("'", "");
      boolean endsByWildcard = false;
      if (pattern.startsWith("*")) {
        sb.append(".*");
        pattern = pattern.substring(1);
      }
      if (pattern.endsWith("*")) {
        endsByWildcard = true;
        pattern = pattern.substring(0, pattern.length() - 1);
      }
      sb.append(Pattern.quote(pattern));
      if (endsByWildcard) {
        sb.append(".*");
      }
      sb.append(",");
    }
    if (sb.length() > 0) {
      sb.delete(sb.length() - 1, sb.length());
    }
    return sb.toString();
  }

  /**
   * <p>Servlet initialization.</p>
   * <p>The first time a SoapUI WsdlProject is created, SoapUI reads its configuration and setup a lot of things.
   * It may be long so let to it during servlet initialization rather at the first use. For the user it will be
   * smoothly.</p>
   * <p>The parent init method from GenericServlet is called.</p>
   *
   * @throws ServletException
   */
  @Override
  public void init() throws ServletException {
    super.init();
    try {
      Misc.fastDynamicLog(LOG, Level.INFO, "Initializing SoapUI.");
      //Sets SoapUI home. SoapUI configuration will be loaded from this directory.
      //We use this to prevent reading a default system-wild configuration
      final File soapUIHome = new File(TMP_DIR + File.separator + "gwt-soapui");
      final File soapUIConfigFile = new File(soapUIHome, SoapUICore.DEFAULT_SETTINGS_FILE);
      if (!soapUIHome.exists()) {
        if (!soapUIHome.mkdirs()) {
          Misc.fastDynamicLog(LOG, Level.WARNING, "Failed to create path '%s'. SoapUI configuration will fall back to" +
              " the default behavior.", soapUIHome.getAbsolutePath());
        } else {
          if (!soapUIConfigFile.exists()) {
            //Create an empty file.
            if (!soapUIConfigFile.createNewFile()) {
              Misc.fastDynamicLog(LOG, Level.WARNING, "Failed to create SoapUI configuration file '%s'." +
                  "SoapUI configuration will fall back to the default behavior.", soapUIConfigFile.getAbsolutePath());
            } else {
              Misc.fastDynamicLog(LOG, Level.FINE, "Custom empty SoapUI configuration file created at '%s'.",
                  soapUIConfigFile.getAbsolutePath());
            }
          }
        }
      }
      SoapUI.setSoapUICore(new DefaultSoapUICore(null, soapUIConfigFile.getAbsolutePath() ), true );

      Misc.fastDynamicLog(LOG, Level.INFO, "Initialization done.");
    } catch (IOException e) {
      throw new ServletException("SoapUI configuration error", e);
    }
  }

  @Override
  public void destroy() {
    SoapUI.shutdown();
    super.destroy();
  }

  /**
   * <p>Retrieves interfaces described in a Wsdl.</p>
   *
   * @param wsdlUrl        URL of the given Wsdl
   * @param createRequests boolean indicating if requests should be created
   * @return {@link WsdlInterface}s described in the given Wsdl
   * @throws ServiceException Gwt-SoapUI wrapper for all SoapUI exceptions
   */
  public synchronized WsdlInterface[] getInterfaces(final String wsdlUrl, final boolean createRequests) throws ServiceException {
    final ClassLoader oldClassLoader = Thread.currentThread().getContextClassLoader();
    Thread.currentThread().setContextClassLoader(XmlObject.class.getClassLoader());
    try {
      if (System.getProperty("http.proxyHost") != null && System.getProperty("http.nonProxyHosts") != null &&
          SoapUI.getSettings().getString(ProxySettings.EXCLUDES, null) == null) {
        //SOAPUI does not check the nonProxyHosts variable so let configure it manually
        Misc.fastDynamicLog(LOG, Level.FINE, "Setting system property 'nonProxyHosts' to SoapUI config.");
        String nonProxyHosts = System.getProperty("http.nonProxyHosts");
        SoapUI.getSettings().setString(ProxySettings.EXCLUDES,
            SoapUIServlet.generateNonProxyHostsInSoapUIFormat(nonProxyHosts));
      }
      WsdlProject project = new WsdlProject();
      Misc.fastDynamicLog(LOG, Level.FINE, "Retrieving interfaces for url '%s.'", wsdlUrl);
      return WsdlInterfaceFactory.importWsdl(project, wsdlUrl, createRequests);
    } catch (XmlException e) {
      Misc.fastDynamicLog(LOG, Level.FINE, "Unable to get wsdl interfaces:\n%s", Misc.getStackTraceFrom(e));
      throw new ServiceException("Unable to get wsdl interfaces", e.getMessage());
    } catch (IOException e) {
      Misc.fastDynamicLog(LOG, Level.FINE, "Unable to get wsdl interfaces:\n%s", Misc.getStackTraceFrom(e));
      throw new ServiceException("Unable to get wsdl interfaces", e.getMessage());
    } catch (SoapUIException e) {
      Misc.fastDynamicLog(LOG, Level.FINE, "Unable to get wsdl interfaces:\n%s", Misc.getStackTraceFrom(e));
      throw new ServiceException("Unable to get wsdl interfaces", e.getMessage());
    } finally {
      Thread.currentThread().setContextClassLoader(oldClassLoader);
    }


  }

  /**
   * <p>Retrieves interfaces described in a Wsdl.</p>
   *
   * @param wsdlUrl URL of the given Wsdl
   * @return Names of interfaces described in the given Wsdl
   * @throws ServiceException Gwt-SoapUI wrapper for all SoapUI exceptions
   */
  public List<String> getInterfaces(final String wsdlUrl) throws ServiceException {
    List<String> interfaces = new ArrayList<String>();
    WsdlInterface[] ifces = this.getInterfaces(wsdlUrl, false);

    for (WsdlInterface ifce : ifces) {
      interfaces.add(ifce.getName());
    }
    return interfaces;
  }

  /**
   * <p>Retrieves operations described in a Wsdl, for a given interface.</p>
   *
   * @param wsdlUrl       URL of the given Wsdl
   * @param interfaceName name of the interface for which operations should be retrieved
   * @return Names of operations described in the given Wsdl
   * @throws ServiceException Gwt-SoapUI wrapper for all SoapUI exceptions
   */
  public List<String> getOperations(final String wsdlUrl, final String interfaceName) throws ServiceException {
    List<String> operations = new ArrayList<String>();
    WsdlInterface[] ifces = this.getInterfaces(wsdlUrl, false);
    for (WsdlInterface ifce : ifces) {
      if (ifce.getName().equals(interfaceName)) {
        for (Operation operation : ifce.getOperationList()) {
          operations.add(operation.getName());
        }
        break;
      }
    }
    return operations;
  }

  /**
   * <p>Generates a request for a given operation.</p>
   *
   * @param wsdlUrl       URL of the given Wsdl
   * @param interfaceName name of the interface for which operations should be retrieved
   * @param operationName name of the operation for which request should be generated
   * @return Request generated for the given operation
   * @throws ServiceException Gwt-SoapUI wrapper for all SoapUI exceptions
   */
  public String generateRequest(final String wsdlUrl, final String interfaceName, final String operationName) throws ServiceException {
    String request = "";
    WsdlInterface[] ifces = this.getInterfaces(wsdlUrl, false);

    for (WsdlInterface ifce : ifces) {
      if (ifce.getName().equals(interfaceName)) {
        WsdlOperation operation = ifce.getOperationByName(operationName);
        request = operation.createRequest(true);
        break;
      }

    }
    return request;
  }

  /**
   * <p>Sends a request for a given operation.</p>
   *
   * @param wsdlUrl       URL of the given Wsdl
   * @param interfaceName name of the interface for which operations should be retrieved
   * @param operationName name of the operation for which request should be generated
   * @param request       the request to send
   * @return Service response
   * @throws ServiceException Gwt-SoapUI wrapper for all SoapUI exceptions
   */
  public String sendRequest(final String wsdlUrl, final String interfaceName, final String operationName, final String request) throws ServiceException {
    String textResponse = "";
    try {
      WsdlInterface[] ifces = this.getInterfaces(wsdlUrl, true);

      for (WsdlInterface ifce : ifces) {
        if (ifce.getName().equals(interfaceName)) {
          WsdlOperation operation = ifce.getOperationByName(operationName);
          WsdlRequest wsdlRequest = operation.addNewRequest(interfaceName + "." + operationName);
          wsdlRequest.setRequestContent(request);
          WsdlSubmitContext wsdlSubmitContext = new WsdlSubmitContext(wsdlRequest);
          WsdlSubmit<WsdlRequest> submit = wsdlRequest.submit(wsdlSubmitContext, false);
          Response response = submit.getResponse();
          textResponse = response.getContentAsString();
          break;
        }

      }
    } catch (Request.SubmitException e) {
      Misc.fastDynamicLog(LOG, Level.FINE, "Unable to send request for operation:\n%s", Misc.getStackTraceFrom(e));
      throw new ServiceException("Unable to send request for operation '" + interfaceName + "." + operationName + "'",
          e.getMessage());
    }
    return textResponse;
  }
}
