/*
 * Copyright (C) 2012 Bull S. A. S.
 * Bull, Rue Jean Jaures, B.P.68, 78340, Les Clayes-sous-Bois
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation
 * version 2.1 of the License.
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA  02110-1301, USA.
 */

package org.ow2.orchestra.common.gwt.utils.client.model;

import com.google.gwt.user.client.rpc.IsSerializable;

/**
 * @author Loic Albertin
 */
public class GwtQName implements IsSerializable {
  public static final String DEFAULT_NS_PREFIX = "";
  public static final String NULL_NS_URI = "";

  /**
   * <p>Namespace URI of this <code>GwtQName</code>.</p>
   */
  private String namespaceURI;

  /**
   * <p>local part of this <code>GwtQName</code>.</p>
   */
  private String localPart;

  /**
   * <p>prefix of this <code>GwtQName</code>.</p>
   */
  private String prefix;

  /**
   * Default no-args constructor for serialization
   */
  public GwtQName() {
  }

  /**
   * <p>GwtQName constructor specifying the Namespace URI
   * and local part.</p>
   *
   * @param namespaceURI Namespace URI of the GwtQName
   * @param localPart    local part of the GwtQName
   */
  public GwtQName(final String namespaceURI, final String localPart) {
    this(namespaceURI, localPart, DEFAULT_NS_PREFIX);
  }

  /**
   * <p><code>QName</code> constructor specifying the Namespace URI,
   * local part and prefix.</p>
   *
   * @param namespaceURI Namespace URI of the <code>QName</code>
   * @param localPart    local part of the <code>QName</code>
   * @param prefix       prefix of the <code>QName</code>
   * @throws IllegalArgumentException When <code>localPart</code>
   *                                  or <code>prefix</code> is <code>null</code>
   */
  public GwtQName(String namespaceURI, String localPart, String prefix) {

    // map null Namespace URI to default
    // to preserve compatibility with QName 1.0
    if (namespaceURI == null) {
      this.namespaceURI = NULL_NS_URI;
    } else {
      this.namespaceURI = namespaceURI;
    }

    // local part is required.
    // "" is allowed to preserve compatibility with QName 1.0
    if (localPart == null) {
      throw new IllegalArgumentException(
          "local part cannot be \"null\" when creating a GwtQName");
    }
    this.localPart = localPart;

    // prefix is required
    if (prefix == null) {
      throw new IllegalArgumentException(
          "prefix cannot be \"null\" when creating a GwtQName");
    }
    this.prefix = prefix;
  }

  /**
   * <p><code>GwtQName</code> constructor specifying the local part.</p>
   *
   * @param localPart local part of the <code>QName</code>
   * @throws IllegalArgumentException When <code>localPart</code> is
   *                                  <code>null</code>
   */
  public GwtQName(String localPart) {
    this(
        NULL_NS_URI,
        localPart,
        DEFAULT_NS_PREFIX);
  }

  /**
   * <p>Get the Namespace URI of this <code>QName</code>.</p>
   *
   * @return Namespace URI of this <code>QName</code>
   */
  public String getNamespaceURI() {
    return namespaceURI;
  }

  /**
   * <p>Get the local part of this <code>QName</code>.</p>
   *
   * @return local part of this <code>QName</code>
   */
  public String getLocalPart() {
    return localPart;
  }

  /**
   * <p>Get the prefix of this <code>QName</code>.</p>
   * <p/>
   * <p>The prefix assigned to a <code>QName</code> might
   * <strong><em>NOT</em></strong> be valid in a different
   * context. For example, a <code>QName</code> may be assigned a
   * prefix in the context of parsing a document but that prefix may
   * be invalid in the context of a different document.</p>
   *
   * @return prefix of this <code>QName</code>
   */
  public String getPrefix() {
    return prefix;
  }

  public final boolean equals(Object objectToTest) {
    if (objectToTest == this) {
      return true;
    }

    if (objectToTest == null || !(objectToTest instanceof GwtQName)) {
      return false;
    }

    GwtQName qName = (GwtQName) objectToTest;

    return localPart.equals(qName.localPart)
        && namespaceURI.equals(qName.namespaceURI);
  }

  public final int hashCode() {
    return namespaceURI.hashCode() ^ localPart.hashCode();
  }

  /**
   * <p><code>String</code> representation of this
   * <code>QName</code>.</p>
   * <p/>
   *
   * @return <code>String</code> representation of this <code>QName</code>
   */
  public String toString() {
    if (namespaceURI.equals(NULL_NS_URI)) {
      return localPart;
    } else {
      return "{" + namespaceURI + "}" + localPart;
    }
  }

  /**
   * <p><code>QName</code> derived from parsing the formatted
   * <code>String</code>.</p>
   *
   * @param qNameAsString <code>String</code> representation
   *                      of the <code>QName</code>
   * @return <code>QName</code> corresponding to the given <code>String</code>
   * @throws IllegalArgumentException When <code>qNameAsString</code> is
   *                                  <code>null</code> or malformed
   */
  public static GwtQName valueOf(String qNameAsString) {

    // null is not valid
    if (qNameAsString == null || "".equals(qNameAsString)) {
      throw new IllegalArgumentException(
          "cannot create QName from \"null\" or \"\" String");
    }

    // local part only?
    if (qNameAsString.charAt(0) != '{') {
      return new GwtQName(
          NULL_NS_URI,
          qNameAsString,
          DEFAULT_NS_PREFIX);
    }

    // Namespace URI improperly specified?
    if (qNameAsString.startsWith("{" + NULL_NS_URI + "}")) {
      throw new IllegalArgumentException(
          "Namespace URI .equals(NULL_NS_URI), .equals(\"" + NULL_NS_URI +
              "\"), only the local part, \""
              + qNameAsString.substring(2 + NULL_NS_URI.length())
              + "\", should be provided.");
    }

    // Namespace URI and local part specified
    int endOfNamespaceURI = qNameAsString.indexOf('}');
    if (endOfNamespaceURI == -1) {
      throw new IllegalArgumentException(
          "cannot create QName from \"" + qNameAsString + "\", missing closing \"}\"");
    }
    return new GwtQName(
        qNameAsString.substring(1, endOfNamespaceURI),
        qNameAsString.substring(endOfNamespaceURI + 1),
        DEFAULT_NS_PREFIX);
  }


  /*
   *****************************
   * Setters for serialization
   *****************************
   */

  public void setNamespaceURI(String namespaceURI) {
    this.namespaceURI = namespaceURI;
  }

  public void setLocalPart(String localPart) {
    this.localPart = localPart;
  }

  public void setPrefix(String prefix) {
    this.prefix = prefix;
  }
}
