/*
 * Copyright (C) 2012 Bull S. A. S.
 * Bull, Rue Jean Jaures, B.P.68, 78340, Les Clayes-sous-Bois
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation
 * version 2.1 of the License.
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA  02110-1301, USA.
 */

package org.ow2.orchestra.common.gwt.utils.client.ui.popup;

import com.google.gwt.core.client.GWT;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.logical.shared.CloseEvent;
import com.google.gwt.event.logical.shared.OpenEvent;
import com.google.gwt.resources.client.ClientBundle;
import com.google.gwt.resources.client.CssResource;
import com.google.gwt.resources.client.ImageResource;
import com.google.gwt.uibinder.client.UiBinder;
import com.google.gwt.uibinder.client.UiField;
import com.google.gwt.uibinder.client.UiHandler;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.DialogBox;
import com.google.gwt.user.client.ui.DisclosurePanel;
import com.google.gwt.user.client.ui.Image;
import com.google.gwt.user.client.ui.Label;
import com.google.gwt.user.client.ui.PushButton;
import com.google.gwt.user.client.ui.TextArea;
import com.google.gwt.user.client.ui.Widget;
import org.ow2.orchestra.common.gwt.utils.client.OrchestraGwtException;

/**
 * @author Loic Albertin
 */
public class Popup extends Composite {
  interface PopupUIBinder extends UiBinder<Widget, Popup> {
  }

  private static PopupUIBinder popupUIBinder = GWT.create(PopupUIBinder.class);

  interface Resources extends ClientBundle {

    @Source("error.png")
    ImageResource errorImage();

    @Source("info.png")
    ImageResource infoImage();

    @Source("close_button.png")
    ImageResource closePushButton();

    @Source("close_button_hover.png")
    ImageResource closePushButtonHover();

    @Source("Popup.css")
    Style popupStyle();
  }

  interface Style extends CssResource {

    String closePushButtonStyle();

    String infoLabelStyle();

    String buttonDeployStyle();

    String infoPopupDialogBoxCaption();

    String errorPopupDialogBoxCaption();

    String headerButtonDeployStyleTd();

  }

  private static final Resources RESOURCES = GWT.create(Resources.class);

  @UiField
  DialogBox dialogBox;
  @UiField
  Image errorImage;
  @UiField
  Image infoImage;
  @UiField
  Label infoLabel;
  @UiField
  DisclosurePanel disclosurePanel;
  @UiField
  TextArea detailsTextArea;
  @UiField
  PushButton closeButton;

  /**
   * Type of the popup.
   *
   * @author Mathieu Crassin
   */
  public enum PopupType {
    ErrorPopup,
    InfoPopup
  }


  private Popup(final PopupType popupType, final String message, final String details) {
    this.initWidget(popupUIBinder.createAndBindUi(this));
    infoLabel.setText(message);
    this.disclosurePanel.getHeader().setStyleName(RESOURCES.popupStyle().buttonDeployStyle());
    if (details != null && !"".equals(details)) {
      this.disclosurePanel.setVisible(true);
      this.disclosurePanel.getHeader().asWidget().addStyleDependentName(RESOURCES.popupStyle().headerButtonDeployStyleTd());
      detailsTextArea.setText(details);
    }
    switch (popupType) {
      case ErrorPopup:
        errorImage.setVisible(true);
        dialogBox.setText("Error!");
        dialogBox.getCaption().asWidget().addStyleName(RESOURCES.popupStyle().errorPopupDialogBoxCaption());
        break;
      case InfoPopup:
        infoImage.setVisible(true);
        dialogBox.setText("Information!");
        dialogBox.getCaption().asWidget().addStyleName(RESOURCES.popupStyle().infoPopupDialogBoxCaption());
        break;
    }
    this.setDebugID();
    this.dialogBox.center();
    RESOURCES.popupStyle().ensureInjected();
  }

  @UiHandler("closeButton")
  public void handleClick(ClickEvent event) {
    this.dialogBox.hide();
  }

  @UiHandler("disclosurePanel")
  public void handleClose(CloseEvent<DisclosurePanel> event) {
    disclosurePanel.getHeaderTextAccessor().setText("More Details");
  }

  @UiHandler("disclosurePanel")
  public void handleOpen(OpenEvent<DisclosurePanel> event) {
    disclosurePanel.getHeaderTextAccessor().setText("Less Details");
  }

  public static void show(final PopupType popupType, final String message, final String details) {
    new Popup(popupType, message, details);
  }

  public static void showError(final String message, final String details) {
    Popup.show(PopupType.ErrorPopup, message, details);
  }

  public static void showError(final String message) {
    Popup.showError(message, null);
  }

  public static void showInfo(final String message, final String details) {
    Popup.show(PopupType.InfoPopup, message, details);
  }

  public static void showInfo(final String message) {
    Popup.showInfo(message, null);
  }

  public static void show(final OrchestraGwtException exception) {
    Popup.showError(exception.getMessage(), exception.getDetails());
  }

  /**
   * Define an id for each widget.
   * This Id is used by selenium for the integration phase.
   *
   * @author Mathieu Crassin
   */
  private void setDebugID() {
    this.dialogBox.ensureDebugId("popupId");
    this.infoLabel.ensureDebugId("popupInfoLabelId");
    this.errorImage.ensureDebugId("popupErrorImageId");
    this.infoImage.ensureDebugId("popupInfoImageId");
    this.detailsTextArea.ensureDebugId("popupDetailsTextareaId");
    this.closeButton.ensureDebugId("popupCloseButtonId");
  }
}
