/*
 * Copyright (C) 2012 Bull S. A. S.
 * Bull, Rue Jean Jaures, B.P.68, 78340, Les Clayes-sous-Bois
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation
 * version 2.1 of the License.
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA  02110-1301, USA.
 */

package org.ow2.orchestra.common.gwt.utils.server;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Properties;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.xml.ws.BindingProvider;
import org.ow2.orchestra.common.gwt.utils.client.OrchestraGwtException;
import org.ow2.orchestra.facade.AccessorUtil;
import org.ow2.orchestra.facade.InstanceManagementAPI;
import org.ow2.orchestra.facade.ManagementAPI;
import org.ow2.orchestra.facade.QueryDefinitionAPI;
import org.ow2.orchestra.facade.QueryRuntimeAPI;
import org.ow2.orchestra.facade.exception.OrchestraException;
import org.ow2.orchestra.util.Misc;
import org.ow2.orchestra.util.OrchestraConstants;
import org.ow2.orchestra.ws_ht.B4PAccessorUtil;
import org.ow2.orchestra.ws_ht.api.wsdl.TaskOperations;

/**
 * Created by: Guillaume Porcher
 * Date: 06/11/11
 * Time: 14:25
 */
public class SessionAPIs {
  private static final Logger LOG = Logger.getLogger(SessionAPIs.class.getName());

    // Default file to list user configurations
    private static final String DEFAULT_USER_CONFIG_FILE = "conf/user-environments.properties";
    // Default environment for deployed services
    private static final String DEFAULT_ORCHESTRA_PROPERTIES_FILE = "conf/orchestra.properties";
    private static final String DEFAULT_ORCHESTRA_PROPERTIES_RESOURCE = "orchestra.properties";

    private static final String ORCHESTRA_PROPERTIES_FILE_PROPERTY = "orchestra.properties.file";
    private static final String ENV_USER_CONFIG_FILE_PROPERTY = "orchestra.user-environments.file";

    /**
   * Service url and Object Name to connect to Orchestra
   */
  protected String serviceUrl = null;
  protected String objectName = null;
  protected String orchestraUrl = null;
  protected Properties orchestraProperties = null;

  private QueryRuntimeAPI queryRuntimeAPI;
  private QueryDefinitionAPI queryDefinitionAPI;
  private InstanceManagementAPI instanceManagementAPI;
  private ManagementAPI managementAPI;
  private TaskOperations taskOperations;

  protected SessionAPIs(String userName) throws OrchestraException, OrchestraGwtException {
    this.initializeOrchestraProperties(userName);
    queryRuntimeAPI = AccessorUtil.getQueryRuntimeAPI(this.serviceUrl, this.objectName);
    queryDefinitionAPI = AccessorUtil.getQueryDefinitionAPI(this.serviceUrl, this.objectName);
    instanceManagementAPI = AccessorUtil.getInstanceManagementAPI(this.serviceUrl, this.objectName);
    managementAPI = AccessorUtil.getManagementAPI(this.serviceUrl, this.objectName);
    getTaskOperationsAPI(userName);
  }

  protected void getTaskOperationsAPI(String userName) {
    if (managementAPI.isExtensionSupported("http://www.example.org/WS-HT")) {
      taskOperations = B4PAccessorUtil.getTaskOperationsAPI(this.orchestraUrl);
      BindingProvider bp = (BindingProvider) taskOperations;
      bp.getRequestContext().put(BindingProvider.USERNAME_PROPERTY, userName);
      bp.getRequestContext().put(BindingProvider.PASSWORD_PROPERTY, userName);
    }
  }

  /**
   * This method is used to get the service url and object name properties from
   * conf file orchestra.properties It first checks if the
   * conf/orchestra.properties exists to use it, otherwise it uses the file
   * included in the war to fill the fields service url and object name of this
   * class
   *
   * @param userName tenant name
   * @throws OrchestraGwtException
   * @author Jean-Yves Goubet
   */
  protected void initializeOrchestraProperties(String userName) throws OrchestraGwtException {
    InputStream inputStream = null;
    final Properties properties = new Properties();
    try {
      // try to load user specific configuration
        final File userEnvFile = new File(System.getProperty(ENV_USER_CONFIG_FILE_PROPERTY, DEFAULT_USER_CONFIG_FILE));
        if (userEnvFile.exists()) {
        final Properties userProperties = new Properties();
        final FileInputStream fip = new FileInputStream(userEnvFile);
        try {
          userProperties.load(fip);
        } finally {
          fip.close();
        }
        if (userName != null && userProperties.containsKey(userName)) {
          String userEnv = userProperties.getProperty(userName);
          File userOrchestraProperties = new File(new File(userEnvFile.getParentFile(), userEnv).getParentFile(), DEFAULT_ORCHESTRA_PROPERTIES_RESOURCE);
          if (userOrchestraProperties.exists()) {
            inputStream = new FileInputStream(userOrchestraProperties);
          }
        }
      }
      // fallback to default configuration file
      if (inputStream == null) {
        final File file = new File(System.getProperty(ORCHESTRA_PROPERTIES_FILE_PROPERTY, DEFAULT_ORCHESTRA_PROPERTIES_FILE));
        if (file.exists()) {
          inputStream = new FileInputStream(file);
        } else {
          // fallbak to embedded configuration
          inputStream = Thread.currentThread().getContextClassLoader().getResourceAsStream(DEFAULT_ORCHESTRA_PROPERTIES_RESOURCE);
        }
      }
      properties.load(inputStream);
      this.orchestraProperties = properties;
      this.serviceUrl = properties.getProperty("orchestra.jmx.serviceUrl");
      this.objectName = properties.getProperty("orchestra.jmx.objectName");
      this.orchestraUrl = "http://" + properties.getProperty(OrchestraConstants.SERVLET_HOST)
          + ":" + properties.getProperty(OrchestraConstants.SERVLET_PORT)
          + "/" + properties.getProperty(OrchestraConstants.SERVLET_PATH);
    } catch (final IOException e) {
      Misc.fastDynamicLog(SessionAPIs.LOG, Level.SEVERE, "OrchestraManagement - getOrchestraProperties :"
          + "Unable to load property file !\n" + Misc.getStackTraceFrom(e));
      throw new OrchestraGwtException("Unable to load property file !",
          "OrchestraManagement - getOrchestraProperties : Unable to load property file !\n" + Misc.getStackTraceFrom(e));
    } finally {
      Misc.close(inputStream);
    }
  }

  public <T> T getAPI(Class<T> clazz) {
    if (clazz.equals(QueryRuntimeAPI.class)) {
      return (T) queryRuntimeAPI;
    } else if (clazz.equals(QueryDefinitionAPI.class)) {
      return (T) queryDefinitionAPI;
    } else if (clazz.equals(ManagementAPI.class)) {
      return (T) managementAPI;
    } else if (clazz.equals(InstanceManagementAPI.class)) {
      return (T) instanceManagementAPI;
    } else if (clazz.equals(TaskOperations.class)) {
      return (T) taskOperations;
    }
    return null;
  }


  public String getOrchestraUrl() {
    return orchestraUrl;
  }

  public Properties getOrchestraProperties() {
    return orchestraProperties;
  }
}
