/*
 * Copyright (C) 2012 Bull S. A. S.
 * Bull, Rue Jean Jaures, B.P.68, 78340, Les Clayes-sous-Bois
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation
 * version 2.1 of the License.
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA  02110-1301, USA.
 */

package org.ow2.orchestra.console.ws.server;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import javax.wsdl.BindingOperation;
import javax.wsdl.Definition;
import javax.wsdl.Input;
import javax.wsdl.Message;
import javax.wsdl.Operation;
import javax.wsdl.Part;
import javax.wsdl.Port;
import javax.wsdl.PortType;
import javax.wsdl.Service;
import javax.wsdl.Types;
import javax.wsdl.extensions.ExtensibilityElement;
import javax.wsdl.extensions.schema.Schema;
import javax.wsdl.extensions.soap.SOAPAddress;
import javax.wsdl.extensions.soap.SOAPHeader;
import javax.xml.namespace.QName;
import javax.xml.soap.SOAPConstants;
import javax.xml.soap.SOAPMessage;
import org.apache.xmlbeans.XmlCursor;
import org.apache.xmlbeans.XmlObject;
import org.apache.xmlbeans.XmlOptions;
import org.ow2.orchestra.common.gwt.utils.client.OrchestraGwtException;
import org.ow2.orchestra.common.gwt.utils.client.model.GwtQName;
import org.ow2.orchestra.common.gwt.utils.server.OrchestraGwtServlet;
import org.ow2.orchestra.console.ws.client.WSConsoleServices;
import org.ow2.orchestra.console.ws.client.model.ActivityEndpoint;
import org.ow2.orchestra.console.ws.client.model.ProcessEndpoints;
import org.ow2.orchestra.console.ws.client.model.SoapRequest;
import org.ow2.orchestra.facade.QueryDefinitionAPI;
import org.ow2.orchestra.facade.def.ActivityDefinition;
import org.ow2.orchestra.facade.def.OnEventDefinition;
import org.ow2.orchestra.facade.def.OnMessageDefinition;
import org.ow2.orchestra.facade.def.PickActivityDefinition;
import org.ow2.orchestra.facade.def.ProcessDefinition;
import org.ow2.orchestra.facade.def.ReceiveActivityDefinition;
import org.ow2.orchestra.facade.def.ScopeActivityDefinition;
import org.ow2.orchestra.facade.exception.ProcessNotFoundException;
import org.ow2.orchestra.facade.uuid.ProcessDefinitionUUID;
import org.ow2.orchestra.util.OrchestraUrlUtil;
import org.ow2.orchestra.util.SOAPUtil;
import org.ow2.orchestra.util.XmlConstants;
import org.ow2.orchestra.util.XmlUtil;
import org.ow2.orchestra.util.wsdl.WSDLInMemoryLocator;
import org.ow2.orchestra.util.wsdl.WsdlInlineUtil;
import org.ow2.orchestra.util.wsdl.WsdlUtil;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

/**
 * @author Loic Albertin
 */
public class WSConsoleServlet extends OrchestraGwtServlet implements WSConsoleServices {
  private static final long serialVersionUID = -1387870563519784245L;

  public ProcessEndpoints getProcessEndpoints(final String processId) throws OrchestraGwtException {
    final QueryDefinitionAPI queryDefinitionAPI = this.getAPI(QueryDefinitionAPI.class);
    final ProcessDefinitionUUID processDefinitionUUID = new ProcessDefinitionUUID(processId);
    final Set<ActivityDefinition> activityDefinitions;
    final ProcessDefinition processDefinition;
    try {
      processDefinition = queryDefinitionAPI.getProcessDefinition(processDefinitionUUID);
      activityDefinitions = queryDefinitionAPI.findActivityDefinitions(processDefinitionUUID);
    } catch (ProcessNotFoundException e) {
      throw new OrchestraGwtException("Process '" + processId + "' not found.", e.getMessage());
    }
    final ProcessEndpoints processEndpoints = new ProcessEndpoints(processId, processDefinition.getName());
    for (ActivityDefinition activityDefinition : activityDefinitions) {
      final String activityDescription = activityDefinition.getDocumentation();
      if (activityDefinition instanceof ReceiveActivityDefinition) {
        ReceiveActivityDefinition receiveActivityDefinition = (ReceiveActivityDefinition) activityDefinition;
        ActivityEndpoint activityEndpoint = new ActivityEndpoint(receiveActivityDefinition.getUUID().toString(),
            receiveActivityDefinition.getName(), activityDescription,
            GwtQName.valueOf(receiveActivityDefinition.getPortType().toString()),
            receiveActivityDefinition.getOperation());
        processEndpoints.addActivityEndpoint(activityEndpoint);
      } else if (activityDefinition instanceof PickActivityDefinition) {
        PickActivityDefinition pickActivityDefinition = (PickActivityDefinition) activityDefinition;
        for (OnMessageDefinition onMessageDefinition : pickActivityDefinition.getOnMessageDefinitionList()) {
          ActivityEndpoint activityEndpoint = new ActivityEndpoint(
              onMessageDefinition.getActivityDefinitionUUID().toString(),
              "OnMessage(" + onMessageDefinition.getOperation() + ")", activityDescription,
              GwtQName.valueOf(onMessageDefinition.getPortType().toString()),
              onMessageDefinition.getOperation());
          processEndpoints.addActivityEndpoint(activityEndpoint);
        }
      } else if (activityDefinition instanceof ScopeActivityDefinition) {
        ScopeActivityDefinition scopeActivityDefinition = (ScopeActivityDefinition) activityDefinition;
        for (OnEventDefinition onEventDefinition : scopeActivityDefinition.getOnEventEventHandlers()) {
          ActivityEndpoint activityEndpoint = new ActivityEndpoint(
              onEventDefinition.getActivityDefinitionUUID().toString(),
              "OnEvent(" + onEventDefinition.getOperation() + ")", activityDescription,
              GwtQName.valueOf(onEventDefinition.getPortType().toString()),
              onEventDefinition.getOperation());
          processEndpoints.addActivityEndpoint(activityEndpoint);
        }
      }
    }
    return processEndpoints;
  }

  @SuppressWarnings("unchecked")
  public SoapRequest generateRequest(final ActivityEndpoint activityEndpoint) throws OrchestraGwtException {
    final String processId = activityEndpoint.getParent().getProcessId();
    final QueryDefinitionAPI queryDefinitionAPI = this.getAPI(QueryDefinitionAPI.class);
    final ProcessDefinitionUUID processDefinitionUUID = new ProcessDefinitionUUID(processId);
    final ProcessDefinition processDefinition;
    try {
      processDefinition = queryDefinitionAPI.getProcessDefinition(processDefinitionUUID);
    } catch (ProcessNotFoundException e) {
      throw new OrchestraGwtException("Process '" + processId + "' not found.", e.getMessage());
    }
    final Collection<String> resourcesName = processDefinition.getProcessResourceNames();
    byte[] bpelContent = null;
    String bpelPath = "";
    final Map<String, byte[]> wsdlFiles = new HashMap<String, byte[]>();

    for (String resource : resourcesName) {
      if (resource.endsWith(".bpel")) {
        if (bpelContent != null) {
          throw new OrchestraGwtException("Repository for process '" + processId +
              "' contains multiples BPEL files. This is not allowed.");
        }
        try {
          if (resource.contains("/")) {
            bpelPath = resource.substring(0, resource.lastIndexOf("/") + 1);
          }
          bpelContent = queryDefinitionAPI.getProcessResource(processDefinitionUUID, resource);
        } catch (ProcessNotFoundException e) {
          throw new OrchestraGwtException("Process '" + processId + "' not found.", e.getMessage());
        }
      } else if (resource.endsWith(".wsdl") || resource.endsWith(".xsd")) {
        try {
          wsdlFiles.put(resource, queryDefinitionAPI.getProcessResource(processDefinitionUUID, resource));
        } catch (ProcessNotFoundException e) {
          throw new OrchestraGwtException("Process '" + processId + "' not found.", e.getMessage());
        }
      }
    }
    if (bpelContent == null) {
      throw new OrchestraGwtException("BPEL definition not found for process '" + processId + "'");
    }

    final List<Schema> schemas = new ArrayList<Schema>();
    final Set<Definition> definitions = new HashSet<Definition>();
    final Document bpelDocument = XmlUtil.getDocumentFromInputStream(new ByteArrayInputStream(bpelContent));
    final NodeList imports = bpelDocument.getElementsByTagNameNS(XmlConstants.XMLNS_BPEL_2_0_EXECUTABLE, "import");
    for (int i = 0; i < imports.getLength(); i++) {
      final Element importElement = (Element) imports.item(i);
      String location = null;
      if (importElement.hasAttribute("location")) {
        location = importElement.getAttribute("location");
      }
      String importType = null;
      if (importElement.hasAttribute("importType")) {
        importType = importElement.getAttribute("importType");
      }
      if (XmlConstants.XMLNS_WSDL.equals(importType)) {
        final Definition definition = WsdlUtil.readWsdl(new WSDLInMemoryLocator(location, bpelPath, wsdlFiles));
        WsdlInlineUtil.inlineWsdlFile(definition);
        definitions.add(definition);
        final Map<String, String> definitionNamespaces = definition.getNamespaces();
        // Process XML Schema imports
        final Types types = definition.getTypes();
        if (types != null) {
          for (final ExtensibilityElement elt : new ArrayList<ExtensibilityElement>(types.getExtensibilityElements())) {
            if (elt instanceof Schema) {
              //Forwards namespaces declared in Definition to schema
              for (Map.Entry<String, String> entry : definitionNamespaces.entrySet()) {
                if (!((Schema) elt).getElement().hasAttribute(
                    "".equals(entry.getKey()) ? "xmlns" : "xmlns:" + entry.getKey())) {
                  ((Schema) elt).getElement().setAttribute(
                      "".equals(entry.getKey()) ? "xmlns" : "xmlns:" + entry.getKey(), entry.getValue());
                }
              }

              schemas.add((Schema) elt);
            }
          }
        }
      }
    }
    return generateSoapRequest(activityEndpoint, schemas, definitions);
  }

  @SuppressWarnings("unchecked")
  private SoapRequest generateSoapRequest(ActivityEndpoint activityEndpoint, List<Schema> schemas, Set<Definition> definitions) throws OrchestraGwtException {
    final String processId = activityEndpoint.getParent().getProcessId();
    for (Definition definition : definitions) {
      final PortType portType = definition.getPortType(QName.valueOf(activityEndpoint.getPortType().toString()));
      if (portType != null) {
        final Operation operation = portType.getOperation(activityEndpoint.getOperationName(), null, null);
        if (operation != null) {
          final BindingOperation bindingOperation = WsdlUtil.getBindingOperation(definition, operation.getName());
          if (bindingOperation == null) {
            //This definition doesn't contains the service implementation
            continue;
          }
          //Retrieve soapAction
          final String soapAction =
              WsdlUtil.getSoapAction(WsdlUtil.getBindingOperation(definition, operation.getName()));
          XmlCursor xmlCursor = null;
          try {
            final boolean isRpc = WsdlUtil.isRpc(definition, bindingOperation);

            final Map<String, String> suggestedPrefix = new HashMap<String, String>();
            suggestedPrefix.put(SOAPConstants.URI_NS_SOAP_1_1_ENVELOPE, "env");
            final XmlOptions options = new XmlOptions().setSaveSuggestedPrefixes(suggestedPrefix);
            final XmlObject soapMessage = XmlObject.Factory.newInstance(options);
            xmlCursor = soapMessage.newCursor();
            //Move into document
            xmlCursor.toNextToken();
            //Create envelope
            xmlCursor.beginElement("Envelope", SOAPConstants.URI_NS_SOAP_1_1_ENVELOPE);
            //Move into Envelope
            xmlCursor.toFirstChild();
            //Create header
            xmlCursor.beginElement("Header", SOAPConstants.URI_NS_SOAP_1_1_ENVELOPE);
            //Move into header
            xmlCursor.toFirstChild();

            final List<ExtensibilityElement> extensibilityElements = bindingOperation.getExtensibilityElements();
            final List<SOAPHeader> soapHeaders = WsdlUtil.getExtensibilityElements(extensibilityElements, SOAPHeader.class);
            for (SOAPHeader soapHeader : soapHeaders) {
              final Message headerMessage = definition.getMessage(soapHeader.getMessage());
              if (headerMessage != null) {
                final List<Part> parts = headerMessage.getOrderedParts(null);
                for (Part part : parts) {
                  XBeansUtils.createPartElement(part, schemas, xmlCursor, options);
                }

              }
            }
            //Move back to Envelope
            xmlCursor.toParent();
            //Jump over Header
            xmlCursor.toLastChild();
            xmlCursor.toEndToken();
            xmlCursor.toNextToken();
            //create body
            xmlCursor.beginElement("Body", SOAPConstants.URI_NS_SOAP_1_1_ENVELOPE);
            //Move into body
            xmlCursor.toFirstChild();


            final Input input = operation.getInput();
            final Message message = input.getMessage();
            if (isRpc) {
              //In RPC parts are contained in an element with the message name
              xmlCursor.beginElement(message.getQName());
              //Move into element
              xmlCursor.toFirstChild();
            }

            final List<Part> parts = message.getOrderedParts(null);
            for (Part part : parts) {
              xmlCursor.push();
              xmlCursor.toLastChild();
              XBeansUtils.createPartElement(part, schemas, xmlCursor, options);
              xmlCursor.pop();
            }

            return new SoapRequest(activityEndpoint, soapAction,
                retrieveEndpointUrl(definitions, portType), XBeansUtils.prettyToString(soapMessage));
          } catch (Exception e) {
            throw new OrchestraGwtException("Process '" + processId + "' unable to build SOAP request.",
                e.getMessage());
          } finally {
            if (xmlCursor != null) {
              xmlCursor.dispose();
            }
          }

        }
      }
    }
    throw new OrchestraGwtException("Process '" + processId + "' unable to build SOAP request.");
  }

  private String retrieveEndpointUrl(final Set<Definition> definitions, final PortType portType) throws OrchestraGwtException {
    String endpointUrlString = "";
    final List<Service> services = WsdlUtil.getServicesOfPortType(portType.getQName(), definitions);
    for (Service service : services) {
      final Map<QName, Port> portsMap = service.getPorts();
      for (Map.Entry<QName, Port> entry : portsMap.entrySet()) {
        if (entry.getValue().getBinding().getPortType().equals(portType)) {
          final SOAPAddress soapAddress =
              WsdlUtil.getExtensibilityElement(entry.getValue().getExtensibilityElements(), SOAPAddress.class);
          final URI endpointUrl;
          try {
            endpointUrl = new URI(OrchestraUrlUtil.replacePropertiesInOrchestraURL(soapAddress.getLocationURI(),
                this.getSessionAPIs().getOrchestraProperties()));
          } catch (URISyntaxException e) {
            continue;
          }
          endpointUrlString = endpointUrl.toString();
        }
      }
    }
    if ("".equals(endpointUrlString)) {
      throw new OrchestraGwtException("Unable to get WS endpoint url from WSDLs.");
    }
    return endpointUrlString;
  }

  public String sendRequest(final SoapRequest soapRequest) throws OrchestraGwtException {
    final ActivityEndpoint activityEndpoint = soapRequest.getActivityEndpoint();
    final String processId = activityEndpoint.getParent().getProcessId();
    final SOAPMessage soapMessage = SOAPUtil.buildSOAPMessage(soapRequest.getSoapAction(), soapRequest.getRequest());

    final SOAPMessage soapResponse = SOAPUtil.call(soapMessage, soapRequest.getEndpointUrl());
    final ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
    try {
      soapResponse.writeTo(outputStream);
    } catch (Exception e) {
      throw new OrchestraGwtException("Process '" + processId + "' retrieve SOAP request.",
          e.getMessage());
    }
    try {
      return XBeansUtils.prettyToString(outputStream.toString());
    } catch (Exception e) {
      throw new OrchestraGwtException("Process '" + processId + "': unable to retrieve SOAP response.", e.getMessage());
    }
  }


}
