/*
 * Copyright (C) 2012 Bull S. A. S.
 * Bull, Rue Jean Jaures, B.P.68, 78340, Les Clayes-sous-Bois
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation
 * version 2.1 of the License.
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA  02110-1301, USA.
 */

package org.ow2.orchestra.console.ws.server;

import java.io.IOException;
import java.io.StringWriter;
import java.io.Writer;
import java.util.List;
import javax.wsdl.Part;
import javax.wsdl.extensions.schema.Schema;
import javax.xml.namespace.QName;
import org.apache.xmlbeans.SchemaGlobalElement;
import org.apache.xmlbeans.SchemaType;
import org.apache.xmlbeans.SchemaTypeLoader;
import org.apache.xmlbeans.SchemaTypeSystem;
import org.apache.xmlbeans.XmlBeans;
import org.apache.xmlbeans.XmlCursor;
import org.apache.xmlbeans.XmlException;
import org.apache.xmlbeans.XmlObject;
import org.apache.xmlbeans.XmlOptions;
import org.ow2.orchestra.common.gwt.utils.client.OrchestraGwtException;
import org.ow2.orchestra.console.ws.server.xsd2inst.SampleXmlUtil;

/**
 * @author Loic Albertin
 */
public class XBeansUtils {
  /**
   * Utility class not designed to be instantiated
   */
  private XBeansUtils() {
  }

  /**
   * Indent an XML string using XmlBeans
   *
   * @param xmlAsString the XML string
   * @return the same XML string with a correct indentation
   * @throws XmlException Throws if the  XML string is not valid
   * @throws IOException
   */
  public static String prettyToString(final String xmlAsString) throws XmlException, IOException {
    final StringWriter stringWriter = new StringWriter();
    XmlObject xmlObject = XmlObject.Factory.parse(xmlAsString);
    XBeansUtils.prettyWrite(xmlObject, stringWriter);
    return stringWriter.toString();
  }

  /**
   * Indent the result of an {@link XmlObject#save(java.io.Writer)} operation.
   *
   * @param xmlObject The XmlObject to convert in string
   * @return The returned string
   * @throws IOException
   */
  public static String prettyToString(final XmlObject xmlObject) throws IOException {
    final StringWriter stringWriter = new StringWriter();
    XBeansUtils.prettyWrite(xmlObject, stringWriter);
    return stringWriter.toString();
  }

  /**
   * Wrapper for the
   * {@link XBeansUtils#prettyWrite(org.apache.xmlbeans.XmlObject, java.io.Writer, org.apache.xmlbeans.XmlOptions)}
   * operation with default options.
   *
   * @param xmlObject The XmlObject to write
   * @param writer    The writer to use
   * @throws IOException
   * @see XBeansUtils#prettyWrite(org.apache.xmlbeans.XmlObject, java.io.Writer, org.apache.xmlbeans.XmlOptions)
   */
  public static void prettyWrite(final XmlObject xmlObject, Writer writer) throws IOException {
    XBeansUtils.prettyWrite(xmlObject, writer, null);
  }

  /**
   * Saves an XmlObject to a writer using options to have a pretty indentation.
   *
   * @param xmlObject The XmlObject to write
   * @param writer    The writer to use
   * @param options   Additional options
   * @throws IOException
   */
  public static void prettyWrite(final XmlObject xmlObject, Writer writer, XmlOptions options) throws IOException {
    if (options == null) {
      options = new XmlOptions();
    }
    options.setSavePrettyPrint().setSavePrettyPrintIndent(2).setSaveNoXmlDecl().setSaveAggressiveNamespaces();
    xmlObject.save(writer, options);
  }

  /**
   * Compile a set of schemas and merge then with the default builtin type system
   * {@link XmlBeans#getBuiltinTypeSystem}.
   *
   * @param schemas The set of schemas to compile
   * @param options Additional options for the schema compilation {@link XmlBeans#compileXsd(org.apache.xmlbeans.XmlObject[], org.apache.xmlbeans.SchemaTypeLoader, org.apache.xmlbeans.XmlOptions)}
   * @return A schema type loader allowing the retrieve types and elements defined in given schemas
   * @throws OrchestraGwtException
   */
  private static SchemaTypeLoader getSchemaTypeLoaderFromSchemas(List<Schema> schemas,
                                                                 XmlOptions options) throws OrchestraGwtException {
    XmlObject[] wsdlObj = new XmlObject[schemas.size()];
    for (int i = 0; i < schemas.size(); i++) {
      try {
        wsdlObj[i] = XmlObject.Factory.parse(schemas.get(i).getElement());
      } catch (XmlException e) {
        throw new OrchestraGwtException("Unable to build SOAP request.",
            e.getMessage());
      }
    }
    try {
      SchemaTypeSystem schemaTypeSystem = XmlBeans.compileXsd(wsdlObj, XmlBeans.getBuiltinTypeSystem(), options);
      return XmlBeans.typeLoaderUnion(new SchemaTypeLoader[]{schemaTypeSystem,
          XmlBeans.getBuiltinTypeSystem()});
    } catch (XmlException e) {
      throw new OrchestraGwtException("Unable to build SOAP request.",
          e.getMessage());
    }
  }

  /**
   * Uses XmlBeans to generate a sample of a {@link Part} element
   *
   * @param part       The part to write
   * @param schemas    schemas defined in the WSDL containing this Part
   * @param partCursor an XmlCursor to write the part element
   * @param options    Additional XmlBeans options
   * @throws OrchestraGwtException
   */
  public static void createPartElement(Part part, List<Schema> schemas, XmlCursor partCursor, XmlOptions options)
      throws OrchestraGwtException {
    SchemaTypeLoader schemaTypeLoader = XBeansUtils.getSchemaTypeLoaderFromSchemas(schemas, options);

    QName elementName = part.getElementName();
    QName typeName = part.getTypeName();

    SchemaType type;
    if (typeName != null) {
      partCursor.beginElement(part.getName());
      type = schemaTypeLoader.findType(typeName);
    } else {
      partCursor.beginElement(elementName);
      SchemaGlobalElement elm = schemaTypeLoader.findElement(elementName);
      type = elm.getType();
    }
    partCursor.toFirstChild();
    new SampleXmlUtil(false).createSampleForType(type, partCursor);
    partCursor.toParent();
  }
}
