/**
 * Copyright (C) 2008 Bull S. A. S.
 * Bull, Rue Jean Jaures, B.P.68, 78340, Les Clayes-sous-Bois
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation
 * version 2.1 of the License.
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA  02110-1301, USA.
 **/

package org.ow2.orchestra.facade;

import java.util.List;

import javax.xml.namespace.QName;

import org.ow2.orchestra.facade.def.ActivityDefinition;
import org.ow2.orchestra.facade.def.ProcessDefinition;
import org.ow2.orchestra.facade.def.ProcessState;
import org.ow2.orchestra.facade.exception.ActivityNotFoundException;
import org.ow2.orchestra.facade.exception.ProcessNotFoundException;
import org.ow2.orchestra.facade.uuid.ActivityDefinitionUUID;
import org.ow2.orchestra.facade.uuid.ProcessDefinitionUUID;

/**
 * Getters on workflow recorded data.<br>
 * Operations in this API applies to definition entities managed by Nova Orchestra
 * such as:<br>
 * process and activities.<br>
 * Returned records are issued from recorded definition informations.
 *
 * @author Guillaume Porcher
 */
public interface QuerierDefinitionAPI {

  /**
   * Returns the list of processes.
   * @return The list of process definition for processes.
   */
  List<ProcessDefinition> findProcessDefinitions();

  /**
   * Returns the list of processes with the given state.
   * @param state: state of processes
   * @return The list of process definition for processes.
   */
  List<ProcessDefinition> findProcessDefinitions(ProcessState state);

  /**
   * Returns the list of processes with the given state and qname.
   * @param processQName qualified name of the process
   * @param state state of processes
   * @return The list of process definition for processes.
   */
  List<ProcessDefinition> findProcessDefinitions(QName processQName, ProcessState state);

  /**
   * Returns the list of processes with the given qname.
   * @param processQName qualified name of the process
   * @param state: state of processes
   * @return The list of process definition for processes.
   */
  List<ProcessDefinition> findProcessDefinitions(QName processQName);
  /**
   * Returns recorded definition data for the given process definition UUID.
   *
   * @param processUUID the uuid of the process definition.
   * @return the process definition
   * @throws ProcessNotFoundException if processUUID cannot be found
   */
  ProcessDefinition getProcessDefinition(ProcessDefinitionUUID processUUID) throws ProcessNotFoundException;

  /**
   * Returns recorded definition data for the given activity definition UUID.
   *
   * @param activityDefinitionUUID the uuid of the process definition.
   * @return the activity definition
   * @throws ActivityNotFoundException if activityDefinitionUUID cannot be found
   */
  ActivityDefinition getActivityDefinition(ActivityDefinitionUUID activityDefinitionUUID) throws ActivityNotFoundException;

}
