/**
 * Copyright (C) 2008 Bull S. A. S.
 * Bull, Rue Jean Jaures, B.P.68, 78340, Les Clayes-sous-Bois
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation
 * version 2.1 of the License.
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA  02110-1301, USA.
 **/

package org.ow2.orchestra.facade;

import java.util.Set;

import org.ow2.orchestra.facade.exception.InstanceNotFoundException;
import org.ow2.orchestra.facade.exception.ProcessNotFoundException;
import org.ow2.orchestra.facade.runtime.ActivityInstance;
import org.ow2.orchestra.facade.runtime.ActivityState;
import org.ow2.orchestra.facade.runtime.ProcessInstance;
import org.ow2.orchestra.facade.uuid.ProcessDefinitionUUID;
import org.ow2.orchestra.facade.uuid.ProcessInstanceUUID;

/**
 * Getters on workflow recorded data.<br>
 * Operations in this API applies to runtime entities managed by Nova Orchestra
 * such as:<br>
 * process instances and activities.<br>
 * Returned records are issued from recorded runtime informations.
 *
 * @author Gael Lalire
 */
public interface QuerierRuntimeAPI {

  /**
   * Returns all records of activity for the given process instance UUID. An
   * empty set is returned if no instance is found.
   *
   * @param processUUID the uuid of the process instance.
   * @return a set of activity instances corresponding to processUUID.
   * @throws InstanceNotFoundException if processUUID cannot be found
   */
  Set<ActivityInstance> findActivityInstances(ProcessInstanceUUID processUUID) throws InstanceNotFoundException;

  /**
   * Returns all records of activity for the given process instance UUID and the
   * given activity state An empty set is returned if no instance is found.
   *
   * @param processUUID the uuid of the process instance.
   * @param activityState the state of activities ("RUNNING" or "RUNNED").
   * @return a set of activity instances corresponding to processUUID and state
   *         'active'.
   * @throws InstanceNotFoundException if processUUID cannot be found
   */
  Set<ActivityInstance> findActivityInstances(ProcessInstanceUUID processUUID, ActivityState activityState)
  throws InstanceNotFoundException;

  /**
   * Returns all records of instance.<br>
   * An empty set is returned if no instance is found.
   *
   * @return a set of all process instances.
   */
  Set<ProcessInstance> findProcessInstances();

  /**
   * Returns all records of instance for the given process definition UUID.<br>
   * An empty set is returned if no instance is found.
   *
   * @param processDefinitionUUID the uuid of the process definition
   * @return a set of process instances corresponding to processDefinitionUUID
   */
  Set<ProcessInstance> findProcessInstances(ProcessDefinitionUUID processDefinitionUUID) throws ProcessNotFoundException;

  /**
   * Returns all records of instance for the given process definition UUID and
   * the given process state. An empty set is returned if no instance is found.
   *
   * @param processDefinitionUUID the uuid of the process definition
   * @param processInstanceState the state of process instances ("RUNNING" or "RUNNED").
   * @return a set of process instances corresponding to processDefinitionUUID
   *         and state 'active'.
   */
  Set<ProcessInstance> findProcessInstances(ProcessDefinitionUUID processDefinitionUUID, ActivityState processInstanceState)
  throws ProcessNotFoundException;

  /**
   * Returns all records of instance for the given process state. An empty set
   * is returned if no instance is found.
   *
   * @param processInstanceState state of process instances ("RUNNING" or "RUNNED").
   * @return a set of process instances corresponding to state 'active'.
   */
  Set<ProcessInstance> findProcessInstances(ActivityState processInstanceState);

}
