/**
 * Copyright (C) 2007  Bull S. A. S.
 * Bull, Rue Jean Jaures, B.P.68, 78340, Les Clayes-sous-Bois
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation
 * version 2.1 of the License.
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA  02110-1301, USA.
 **/
package org.ow2.orchestra.facade.runtime.impl;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;

import org.ow2.orchestra.facade.runtime.Message;
import org.ow2.orchestra.util.XmlUtil;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

/**
 * Models a WSDL message type variable.
 *
 * @author Goulven Le Jeune, Charles Souillard, Candy Chlad, Stanislas Giraudet
 *         De Boudemange, Guillaume Porcher
 */
public class MessageImpl implements Serializable, Message {

  private static final long serialVersionUID = 6047054200032920537L;

  protected Map<String, String> parts;

  /**
   * Create a multi-part message.
   *
   * @param parts
   *            HashMap containing the message parts
   */
  public MessageImpl(final Map<String, Document> docParts) {
    this.parts = new HashMap<String, String>();
    if (docParts != null) {
      for (final Entry<String, Document> docPart : docParts.entrySet()) {
        this.parts.put(docPart.getKey(), XmlUtil.toString(docPart.getValue()));
      }
    }
  }

  /* (non-Javadoc)
   * @see org.ow2.orchestra.facade.runtime.impl.Message#getPartValue(java.lang.String)
   */
  public Element getPartValue(final String partName) {
    if (partName != null) {
      if (this.parts.get(partName) != null) {
        return XmlUtil.getDocumentFromString(this.parts.get(partName)).getDocumentElement();
      }
      return null;
    }
    return XmlUtil.getDocumentFromString(this.parts.values().iterator().next()).getDocumentElement();
  }

  /**
   * Return true if the part having the given name is initialized.
   *
   * @param partName
   *            partName
   * @return true if the part having the given name is initialized.
   */
  public boolean isPartInitialized(final String partName) {
    return this.parts.get(partName) != null;
  }

  /* (non-Javadoc)
   * @see org.ow2.orchestra.facade.runtime.impl.Message#hasPart(java.lang.String)
   */
  public boolean hasPart(final String partName) {
    if (partName != null) {
      return this.parts.containsKey(partName);
    }
    return false;
  }

  /**
   * Return a string representation of the message.
   *
   * @return a String representation of the message
   */
  @Override
  public String toString() {
    final StringBuffer buf = new StringBuffer();
    buf.append("<message>");
    for (final String part : this.parts.keySet()) {
      if (this.parts.get(part) != null) {
        buf.append("<" + part + ">");
        if (this.parts.get(part) == null) {
          buf.append("no_value_for_this_part");
        } else {
          buf.append(this.parts.get(part));
        }
        buf.append("</" + part + ">");
      }
    }
    buf.append("</message>");
    return buf.toString();
  }

  /* (non-Javadoc)
   * @see org.ow2.orchestra.facade.runtime.impl.Message#getParts()
   */
  public Map<String, Element> getParts() {
    final Map<String, Element> res = new HashMap<String, Element>();
    if (this.parts != null) {
      for (final Map.Entry<String, String> entry : this.parts.entrySet()) {
        if (entry.getValue() != null) {
          res.put(entry.getKey(), XmlUtil.getDocumentFromString(entry.getValue()).getDocumentElement());
        } else {
          res.put(entry.getKey(), null);
        }
      }
    }
    return res;
  }

  /**
   * Export to log message.
   *
   * @return The duplicate of Xml object
   */
  public MessageImpl copy() {
    if (this.parts == null) {
      return null;
    }
    final Map<String, String> duplicateMessageParts = new HashMap<String, String>(this.parts);
    final MessageImpl newMsg = new MessageImpl(null);
    newMsg.parts = duplicateMessageParts;
    return newMsg;
  }

}
