/**
 * Bull SAS / OW2 Orchestra
 * Copyright (C) 2007 Bull S.A.S, and individual contributors as indicated
 * by the @authors tag.
 *
 * Bull, Rue Jean Jaures, B.P.68, 78340, Les Clayes-sous-Bois
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation
 * version 2.1 of the License.
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA  02110-1301, USA.
 **/
package org.ow2.orchestra.util;

import java.io.InputStream;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.net.URI;
import java.net.URL;
import java.util.List;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.wsdl.Definition;
import javax.xml.parsers.DocumentBuilderFactory;

import org.ow2.orchestra.facade.Deployment;
import org.ow2.orchestra.facade.exception.OrchestraRuntimeException;
import org.ow2.orchestra.facade.wsdl.WsdlDefinitions;
import org.ow2.orchestra.util.wsdl.WsdlUtil;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

/**
*
* @author Goulven Le Jeune
*
*/
public final class DeploymentUtil {

  private DeploymentUtil() { }

  private static final Logger LOG = Logger.getLogger(DeploymentUtil.class.getName());
  private static DocumentBuilderFactory documentBuilderFactory = null;

  private static DocumentBuilderFactory getDocumentBuilderFactory() {
    if (DeploymentUtil.documentBuilderFactory == null) {
      DeploymentUtil.documentBuilderFactory = DocumentBuilderFactory.newInstance();
      DeploymentUtil.documentBuilderFactory.setNamespaceAware(true);
    }
    return DeploymentUtil.documentBuilderFactory;
  }

  public static Deployment generateDeployment(final URL bpelUrl, final List<URL> wsdls) {

    if (DeploymentUtil.LOG.isLoggable(Level.INFO)) {
      DeploymentUtil.LOG.info("Parsing BPEL from URL: " + bpelUrl + " using WSDL list: " + wsdls);
    }
    final Deployment deployment = new Deployment();
    deployment.setBpelUrl(bpelUrl);
    final WsdlDefinitions wsdlDefinitions = new WsdlDefinitions();

    if (wsdls != null) {
      for (final URL wsdl : wsdls) {
        final Definition wsdlDef = WsdlUtil.readWsdl(wsdl);
        wsdlDefinitions
            .addWsdlDefinition(wsdlDef.getTargetNamespace(), wsdlDef);
      }
    }

    // Parse Bpel document
    Document document = null;
    try {
      final InputStream inStream = bpelUrl.openStream();
      document = XmlUtil.getDocumentFromInputStream(inStream);
    } catch (final Exception e) {
      throw new OrchestraRuntimeException("couldn't parse xml document", e);
    }
    final NodeList imports = document.getElementsByTagNameNS(XmlConstants.XMLNS_BPEL_2_0_EXECUTABLE, "import");

    final StringWriter stringWriter = new StringWriter();
    final PrintWriter printWriter = new PrintWriter(stringWriter);

    for (int i = 0; i < imports.getLength(); i++) {
      DeploymentUtil.parseImportElement((Element) imports.item(i), printWriter, bpelUrl,
          wsdlDefinitions, deployment);
    }

    printWriter.close();
    final String errorMsg = stringWriter.getBuffer().toString();

    if (!"".equals(errorMsg)) {
      throw new OrchestraRuntimeException("errors during parsing of bpel imports: " + errorMsg);
    }

    // parse stylesheets
    for (final String tagName : new String[]{"from", "to"}) {
      final NodeList nodes = document.getElementsByTagNameNS(XmlConstants.XMLNS_BPEL_2_0_EXECUTABLE, tagName);
      for (int i = 0; i < nodes.getLength(); i++) {
        DeploymentUtil.parseStyleSheets((Element) nodes.item(i), printWriter, bpelUrl, deployment);
      }
    }

    return deployment;
  }



  /**
   * Parse Bpel process import element.
   */
  private static void parseImportElement(final Element importElement, final PrintWriter printWriter,
      final URL bpelURL, final WsdlDefinitions definitions, final Deployment deployment) {
    String namespace = null;
    if (importElement.hasAttribute("namespace")) {
      namespace = importElement.getAttribute("namespace");
    }
    String location = null;
    if (importElement.hasAttribute("location")) {
      location = importElement.getAttribute("location");
    }

    String importType = null;
    if (importElement.hasAttribute("importType")) {
      importType = importElement.getAttribute("importType");
    }

    if (importType == null) {
      printWriter.println("missing importType attribute in import element "
          + XmlUtil.toString(importElement));
    } else if (importType.equals(XmlConstants.XMLNS_WSDL)) {
      // Import a WSDL
      if (location != null) {
        URL wsdlURL = null;
        try {
          final URI bpelURI = new URI(bpelURL.toString().replace(" ", "%20"));
          final String bpelPath = bpelURL.toExternalForm();

          // build url relative to the bpel url
          if (location.startsWith("http://") || location.startsWith("https://")
              || location.startsWith("file:")) {
            wsdlURL = new URL(location);
          } else if (bpelURI.isOpaque() && bpelPath.startsWith("jar:")) {
            final int lastSlash = bpelPath.lastIndexOf("!") + 1;
            final String packageBase = bpelPath.substring(0, lastSlash);
            final String bpelBase = bpelPath.substring(lastSlash);

            final URI wsdlLocationInPackage = URI.create(bpelBase).resolve(location);

            wsdlURL = new URL(packageBase + wsdlLocationInPackage);
          } else {
            wsdlURL = bpelURI.resolve(location).toURL();
          }
        } catch (final Exception e) {
          throw new OrchestraRuntimeException("Exception while building wsdlURL : ", e);
        }
        // parse definition
        final Definition wsdlDefinition = WsdlUtil.readWsdl(wsdlURL);

        deployment.addWsdlDefinition(namespace, wsdlDefinition);
      } else {
        Set<Definition> definitionsForNS = null;
        if (definitions != null) {
          definitionsForNS = definitions.getWsdlDefinitions(namespace);
        }
        if (definitionsForNS == null) {
          printWriter.println("No wsdl file to import in import element "
              + XmlUtil.toString(importElement));
        } else {
          for (final Definition def : definitionsForNS) {
            deployment.addWsdlDefinition(namespace, def);
          }
        }
      }
    } else {
      printWriter.println("unsupported importType attribute in import element " + XmlUtil.toString(importElement));
    }
  }

  /**
   * @param item
   * @param printWriter
   * @param bpelUrl
   * @param deployment
   */
  private static void parseStyleSheets(final Element item, final PrintWriter printWriter, final URL bpelUrl, final Deployment deployment) {
    if (!XmlUtil.isTextOnly(item)) {
      return;
    }
    final String expression = XmlUtil.getContentText(item);
    final String functionNameBegin = "doXslTransform(";
    if (expression.contains(functionNameBegin)) {
      final int indexBegin = expression.indexOf(functionNameBegin);
      final int indexEnd = expression.indexOf(",", indexBegin + functionNameBegin.length());
      final String styleSheet = expression.substring(indexBegin + functionNameBegin.length() + 1, indexEnd - 1);
      URL xslURL = null;
      try {
        if (styleSheet.startsWith("http://") || styleSheet.startsWith("file:")) {
          xslURL = new URL(styleSheet);
        } else {
          final String bpelPath = bpelUrl.toExternalForm();
          final int lastSlash = bpelPath.lastIndexOf("/");
          final int lastBackSlash = bpelPath.lastIndexOf("\\");
          String bpelBase = null;
          if (lastSlash > lastBackSlash) {
            bpelBase = bpelPath.substring(0, lastSlash);
          } else {
            bpelBase = bpelPath.substring(0, lastBackSlash);
          }

          xslURL = new URL(bpelBase + "/" + styleSheet);
        }
      } catch (final Exception e) {
        throw new OrchestraRuntimeException("Exception while building xslURL : ", e);
      }
      final StyleSheetRepository styleSheetRepository = deployment.getStyleSheetRepository();
      styleSheetRepository.addStyleSheet(xslURL, styleSheet);
    }
  }
}
