/**
 * Copyright (C) 2006  Bull S. A. S.
 * Bull, Rue Jean Jaures, B.P.68, 78340, Les Clayes-sous-Bois
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation
 * version 2.1 of the License.
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA  02110-1301, USA.
 **/
package org.ow2.orchestra.util;

import java.io.Serializable;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;

import org.ow2.orchestra.facade.exception.OrchestraRuntimeException;

/**
 *
 * @author Charles Souillard
 * @author Guillaume Porcher
 */
public class ProcessResourcesRepository implements Serializable {


  private static final long serialVersionUID = -67804473770091529L;

  /**
   * Store all the resources used in the process.
   * key: String =>name of the resource
   * value: byte[] => the resource content
   */
  protected Map<String, byte[]> resources = new HashMap<String, byte[]>();

  /**
   * @return the resources
   */
  public Map<String, byte[]> getResources() {
    return this.resources;
  }


  /**
   * Add a new process resource.
   * @param url url of the resource
   * @param resourceName name of the resource
   */
  public void addResource(final URL url, final String resourceName) {
    try {
      final byte[] resourceBytes = Misc.getAllContentFrom(url);
      this.resources.put(resourceName, resourceBytes);
    } catch (final Exception e) {
      throw new OrchestraRuntimeException(
          "Problem when getting resource from url : " + url, e);
    }
  }


  public ClassLoader getResourcesClassLoader(final ClassLoader parentClassLoader) {
    return new ResourceClassLoader(this.resources, parentClassLoader);
  }

  private static class ResourceClassLoader extends ClassLoader {

    private final Map<String, byte[]> resources;

    public ResourceClassLoader(final Map<String, byte[]> resources, final ClassLoader parentClassLoader) {
      super(parentClassLoader);
      this.resources = resources;
    }

    @Override
    public Class< ? > findClass(final String name) throws ClassNotFoundException {
      final String resourceName = name.replace(".", "/") + ".class";
      final byte[] b = this.resources.get(resourceName);
      if (b != null) {
        return this.defineClass(name, b, 0, b.length);
      }
      return super.findClass(name);
    }

  }
}
