/**
 * Copyright (C) 2009 Bull S. A. S.
 * Bull, Rue Jean Jaures, B.P.68, 78340, Les Clayes-sous-Bois
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation
 * version 2.1 of the License.
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA  02110-1301, USA.
 **/

package org.ow2.orchestra.facade;

import java.util.List;

import org.ow2.orchestra.facade.data.runtime.JobData;
import org.ow2.orchestra.facade.exception.InstanceNotFoundException;
import org.ow2.orchestra.facade.exception.JobNotFoundException;
import org.ow2.orchestra.facade.uuid.ProcessInstanceUUID;

/**
 * Manage a process instance.
 *
 * @author Guillaume Renault
 */
public interface InstanceManagementAPI {

  /**
   * Exit the given instance.
   *
   * @param processInstanceUUID
   *          the Id of the instance to exit.
   * @throws InstanceNotFoundException
   *           thrown if the given instance to exit is not found.
   */
  void exit(final ProcessInstanceUUID processInstanceUUID) throws InstanceNotFoundException;


  /**
   * Resume a previously suspend instance of a process.
   *
   * @param processInstanceUUID
   *          the process instance UUID that identify the instance to resume.
   * @throws InstanceNotFoundException
   *           thrown if the given instance to resume is not found.
   */
  void resume(final ProcessInstanceUUID processInstanceUUID) throws InstanceNotFoundException;

  /**
   * Suspend the given instance of a process.
   *
   * @param processInstanceUUID
   *          the process instance UUID that identify the instance to suspend.
   * @throws InstanceNotFoundException
   *           thrown if the given instance to suspend is not found.
   */
  void suspend(final ProcessInstanceUUID processInstanceUUID) throws InstanceNotFoundException;

  /**
   * To delete monitoring information on the process instance.
   * @param processInstanceUUID
   *          the process instance UUID that identify the instance to suspend.
   * @return true if the instance has been successfully deleted otherwise
   * returns false.
   */
  boolean remove(ProcessInstanceUUID processInstanceUUID);

  /**
   * Get the list of asynchronous executions for which all the tries have failed and which will not be executed any more.
   */
  List<JobData> findJobsWithException();

  /**
   * Reschedules a job (asynchronous execution).
   * The job executor will try to execute the job at most retries times.
   * If the job fails for each retry, then the job executor will not try to execute it
   * unless this method is called again for this job.
   * @param job the job to retry.
   * @param retries the number of retries to set for the job.
   * @throws JobNotFoundException if job does not exist.
   */
  void retryJob(JobData job, int retries) throws JobNotFoundException;
}
