/**
 * Copyright (C) 2010 Bull S. A. S.
 * Bull, Rue Jean Jaures, B.P.68, 78340, Les Clayes-sous-Bois
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation
 * version 2.1 of the License.
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA  02110-1301, USA.
 **/

package org.ow2.orchestra.facade;

import java.util.Set;

import org.ow2.orchestra.facade.data.def.ActivityDefinitionData;
import org.ow2.orchestra.facade.data.def.ProcessDefinitionData;
import org.ow2.orchestra.facade.data.runtime.ProcessInstanceData;
import org.ow2.orchestra.facade.exception.InstanceNotFoundException;
import org.ow2.orchestra.facade.exception.ProcessNotFoundException;
import org.ow2.orchestra.facade.uuid.ProcessDefinitionUUID;
import org.ow2.orchestra.facade.uuid.ProcessInstanceUUID;

/**
 * Get information about processes and instances from the engine repository.<br>
 * This API can be used even when monitoring is disabled.
 *
 * @author Guillaume Porcher
 */
public interface RuntimeAPI {

  // DEFINITION

  /**
   * @return the list of process currently in the engine repository.
   */
  Set<ProcessDefinitionData> getProcesses();

  /**
   * Get information about a process.
   * @param processUUID UUID of the process
   * @return information on process with UUID processUUID
   * @throws ProcessNotFoundException if no process with the given UUID exists in the repository.
   */
  ProcessDefinitionData getProcess(ProcessDefinitionUUID processUUID) throws ProcessNotFoundException;

//  /**
//   * Get information about a process activity.
//   * @param activityDefinitionUUID UUID of the activity
//   * @return information on activity with UUID activityDefinitionUUID
//   * @throws ActivityNotFoundException if no activity with the given UUID exists in the repository.
//   */
//  ActivityDefinitionData getActivityDefinition(ActivityDefinitionUUID activityDefinitionUUID) throws ActivityNotFoundException;

  /**
   * Get information about a process activities.
   * @param processUUID UUID of the process
   * @return a list of information on activities of the process with UUID processUUID
   * @throws ProcessNotFoundException if no process with the given UUID exists in the repository.
   */
  Set<ActivityDefinitionData> getActivityDefinitions(ProcessDefinitionUUID processUUID) throws ProcessNotFoundException;


  // INSTANCES

  /**
   * @return the list of process instances currently in the engine repository.
   */
  Set<ProcessInstanceData> getProcessInstances();

  /**
   * Get information about the instances of a process
   * @param processDefinitionUUID UUID of the process
   * @return a list of information on instances of the process with UUID processDefinitionUUID
   * @throws ProcessNotFoundException if no process with the given UUID exists in the repository.
   */
  Set<ProcessInstanceData> getProcessInstances(ProcessDefinitionUUID processDefinitionUUID) throws ProcessNotFoundException;

//  /**
//   * Get information about the instances of a process (filtered by state)
//   * @param processDefinitionUUID UUID of the process
//   * @param processInstanceState state of the instances to return.
//   * @return a list of information on instances of the process with UUID processDefinitionUUID,
//   * which are in processInstanceState state.
//   * @throws ProcessNotFoundException if no process with the given UUID exists in the repository.
//   */
//  Set<ProcessInstanceData> getProcessInstances(ProcessDefinitionUUID processDefinitionUUID, ActivityState processInstanceState)
//    throws ProcessNotFoundException;

//  /**
//   * Get information about the process instances (filtered by state)
//   * @param processInstanceState state of the instances to return.
//   * @return a list of information on instances which are in processInstanceState state.
//   */
//  Set<ProcessInstanceData> getProcessInstances(ActivityState processInstanceState);

  /**
   * Get information about a process instance.
   * @param processInstanceUUID UUID of the instance
   * @return information on process instance with UUID processInstanceUUID
   * @throws InstanceNotFoundException if no process instance with the given UUID exists in the repository.
   */
  ProcessInstanceData getProcessInstance(ProcessInstanceUUID processInstanceUUID) throws InstanceNotFoundException;

//  /**
//   * Get information about a process instance activities.
//   * @param processUUID UUID of the instance
//   * @return a list of information on activities of the process instance with UUID processUUID
//   * @throws InstanceNotFoundException if no process instance with the given UUID exists in the repository.
//   */
//  Set<ActivityInstanceData> getRunningActivityInstances(ProcessInstanceUUID processUUID) throws InstanceNotFoundException;
//
//  /**
//   * Get information about a process instance activity.
//   * @param activityInstanceUUID UUID of the instance activity
//   * @return information on activity instance with UUID activityInstanceUUID
//   * @throws ActivityInstanceNotFoundException if no process instance activity with the given UUID exists in the repository.
//   */
//  ActivityInstanceData getActivityInstance(ActivityInstanceUUID activityInstanceUUID) throws ActivityInstanceNotFoundException;
}
