package org.ow2.orchestra.facade;

import java.io.ByteArrayInputStream;
import java.io.Serializable;
import java.net.URL;

import javax.wsdl.Definition;

import org.ow2.orchestra.facade.wsdl.WsdlDefinitions;
import org.ow2.orchestra.util.Misc;
import org.ow2.orchestra.util.ProcessResourcesRepository;
import org.xml.sax.InputSource;

/**
 * bpel XML process source plus related WSDL XML sources and optional files.
 *
 * bpel deployments can be build from a zip file with
 * {@link #addZipEntries(java.util.zip.ZipInputStream)} or from resources with
 * {@link #addWsdlResource(String)} and {@link #setBpelResource(String)}.
 *
 * @author Tom Baeyens
 */
public class Deployment implements Serializable {

  /**
   *
   */
  private static final long serialVersionUID = -5531271839683239304L;

  /** bpel file as an inputSource. */
  private byte[] bpel;

  /** wsdl file describing the process as a webservice. */
  private WsdlDefinitions wsdlDefinitions;

  /** list of resources (xsl templates) used in the process */
  private final ProcessResourcesRepository resourcesRepository = new ProcessResourcesRepository();

  /**
   * add a wsdl in wsdls map.
   *
   * @param inputSource
   *            wsdl to add
   */
  public void addWsdlDefinition(final String namespace, final Definition wsdlDefinition) {
    if (this.wsdlDefinitions == null) {
      this.wsdlDefinitions = new WsdlDefinitions();
    }
    this.wsdlDefinitions.addWsdlDefinition(namespace, wsdlDefinition);
  }

  /**
   * set the bpel based on an URL.
   *
   * @param url
   *            url to the bpel file
   */
  public void setBpelUrl(final URL url) {
    Misc.checkArgsNotNull(url);
    try {
      this.bpel = Misc.getAllContentFrom(url);
    } catch (final Exception e) {
      throw new RuntimeException("couldn't read bpel resource from url "
          + url, e);
    }
  }

  public InputSource getBpel() {
    return new InputSource(new ByteArrayInputStream(this.bpel));
  }

  public WsdlDefinitions getWsdlDefinitions() {
    return this.wsdlDefinitions;
  }

  /**
   * @return the resources repository
   */
  public ProcessResourcesRepository getResourcesRepository() {
    return this.resourcesRepository;
  }

}
