package org.ow2.orchestra.facade.criteria;

import org.ow2.orchestra.util.JAXBUtils;

import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlElementWrapper;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.adapters.XmlJavaTypeAdapter;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

/**
 * A Restriction represents a constraint that must be observed to match a {@link Criteria}. <br>
 * A Restriction is composed by;
 * <ul>
 * <li>a {@link FilterField} on which the Restriction applies</li>
 * <li>a value on which the Restriction applies</li>
 * <li>an operation that links the field and the value</li>
 * </ul>
 * <p/>
 * Currently supported operations are:
 * <ul>
 * <li>{@link Restriction#EQUALS}</li>
 * <li>{@link Restriction#LESS_THAN}</li>
 * <li>{@link Restriction#LESS_THAN_EQUALS}</li>
 * <li>{@link Restriction#GREATER_THAN}</li>
 * <li>{@link Restriction#GREATER_THAN_EQUALS}</li>
 * <li>{@link Restriction#BETWEEN}</li>
 * </ul>
 *
 * @author Loic Albertin
 */
@XmlRootElement(name = "restriction")
public class Restriction<T> implements Serializable {

  public static final long serialVersionUID = -8141907675530673225L;

  /**
   * Represents an "equality" constraint
   */
  public static final String EQUALS = "=";

  /**
   * Represents an "between" constraint
   */
  public static final String BETWEEN = "><";

  /**
   * Represents an "less than" constraint
   */
  public static final String LESS_THAN = "lt";

  /**
   * Represents an "less than or equals" constraint
   */
  public static final String LESS_THAN_EQUALS = "le";

  /**
   * Represents an "greater than" constraint
   */
  public static final String GREATER_THAN = "gt";

  /**
   * Represents an "greater than or equals" constraint
   */
  public static final String GREATER_THAN_EQUALS = "ge";

  /**
   * The field on which the restriction applies
   */
  @XmlElement
  private FilterField<T> field;

  /**
   * The value on which the Restriction applies
   */
  @XmlElementWrapper
  @XmlJavaTypeAdapter(JAXBUtils.JAXBSerializableAdapter.class)
  private List<Serializable> values;

  /**
   * The operation that links the field and the value
   */
  @XmlElement
  private String operation;

  private Restriction() {}
  /**
   * This constructor is defined as private to for usage of static constructor methods
   *
   * @param field     The field on which the restriction applies
   * @param values    Values on which the Restriction applies
   * @param operation The operation that links the field and the value
   */
  private Restriction(FilterField<T> field, List<Serializable> values, String operation) {
    this.field = field;
    this.values = values;
    this.operation = operation;
  }

  /**
   * @return The field on which the restriction applies
   */
  public final FilterField<T> getField() {
    return field;
  }

  /**
   * @return The value on which the Restriction applies
   */
  public final List getValues() {
    return values;
  }

  /**
   * @return The operation that links the field and the value
   */
  public final String getOperation() {
    return operation;
  }

  /**
   * Static method used to construct an "equality" restriction on a given field and value
   *
   * @param field The field on which the restriction applies
   * @param value The value on which the Restriction applies
   * @return The corresponding {@link Restriction}
   */
  public static <C> Restriction<C> eq(final FilterField<C> field, final Serializable value) {
    final List<Serializable> values = new ArrayList<Serializable>();
    values.add(value);
    return new Restriction<C>(field, values, Restriction.EQUALS);
  }

  /**
   * Static method used to construct an "between" restriction on a given field and lower/upper values
   *
   * @param field The field on which the restriction applies
   * @param lower The lower value on which the Restriction applies
   * @param upper The upper value on which the Restriction applies
   * @return The corresponding {@link Restriction}
   */
  public static <C> Restriction<C> between(final FilterField<C> field, final Serializable lower, final Serializable upper) {
    final List<Serializable> values = new ArrayList<Serializable>();
    values.add(lower);
    values.add(upper);
    return new Restriction<C>(field, values, Restriction.BETWEEN);
  }

  /**
   * Static method used to construct an "less than" restriction on a given field and value
   *
   * @param field The field on which the restriction applies
   * @param value The value on which the Restriction applies
   * @return The corresponding {@link Restriction}
   */
  public static <C> Restriction<C> lt(final FilterField<C> field, final Serializable value) {
    final List<Serializable> values = new ArrayList<Serializable>();
    values.add(value);
    return new Restriction<C>(field, values, Restriction.LESS_THAN);
  }

  /**
   * Static method used to construct an "less than or equals" restriction on a given field and value
   *
   * @param field The field on which the restriction applies
   * @param value The value on which the Restriction applies
   * @return The corresponding {@link Restriction}
   */
  public static <C> Restriction<C> le(final FilterField<C> field, final Serializable value) {
    final List<Serializable> values = new ArrayList<Serializable>();
    values.add(value);
    return new Restriction<C>(field, values, Restriction.LESS_THAN_EQUALS);
  }

  /**
   * Static method used to construct an "greater than" restriction on a given field and value
   *
   * @param field The field on which the restriction applies
   * @param value The value on which the Restriction applies
   * @return The corresponding {@link Restriction}
   */
  public static <C> Restriction<C> gt(final FilterField<C> field, final Serializable value) {
    final List<Serializable> values = new ArrayList<Serializable>();
    values.add(value);
    return new Restriction<C>(field, values, Restriction.GREATER_THAN);
  }

  /**
   * Static method used to construct an "greater than or equals" restriction on a given field and value
   *
   * @param field The field on which the restriction applies
   * @param value The value on which the Restriction applies
   * @return The corresponding {@link Restriction}
   */
  public static <C> Restriction<C> ge(final FilterField<C> field, final Serializable value) {
    final List<Serializable> values = new ArrayList<Serializable>();
    values.add(value);
    return new Restriction<C>(field, values, Restriction.GREATER_THAN_EQUALS);
  }

}
