/**
 * Copyright (C) 2011  Bull S. A. S.
 * Bull, Rue Jean Jaures, B.P.68, 78340, Les Clayes-sous-Bois
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation
 * version 2.1 of the License.
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA  02110-1301, USA.
 **/
package org.ow2.orchestra.facade.criteria;

import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlRootElement;
import java.io.Serializable;

/**
 * An Order order represent a sort on a given field. It could be either in
 * ascending or in descending direction.
 *
 * @author Loic Albertin
 *
 */
@XmlRootElement(name = "order")
public class Order<T> implements Serializable {

  private static final long serialVersionUID = -8141908789530673225L;

  /**
   * The field on which the sort applies
   */
  @XmlElement
  private SortableField<T> field;

  /**
   * Represent the sort direction. Default is ascending.
   */
  @XmlElement
  private boolean ascending = true;

  private Order() {}
  /**
   * @param field The field on which the sort applies
   * @param ascending Represent the sort direction, true is ascending, false is
   *          descending.
   */
  public Order(final SortableField<T> field, final boolean ascending) {
    super();
    this.field = field;
    this.ascending = ascending;
  }

  /**
   * @return the field The field on which the sort applies
   */
  public final SortableField<T> getField() {
    return field;
  }

  /**
   * @return the ascending Represent the sort direction, true is ascending,
   *         false is descending.
   */
  public final boolean isAscending() {
    return ascending;
  }

  // ************************************************
  // ******** Static constructor methods ************
  // ************************************************

  /**
   * Creates an ascending order on the given sortable field
   *
   * @param field the field to sort
   * @return An ascending order on the given field
   */
  public static <U> Order<U> ascending(final SortableField<U> field) {
    return new Order<U>(field, true);
  }

  /**
   * Creates an descending order on the given sortable field
   *
   * @param field the field to sort
   * @return An descending order on the given field
   */
  public static <U> Order<U> descending(final SortableField<U> field) {
    return new Order<U>(field, false);
  }

}
