/**
 * Copyright (C) 2008 Bull S. A. S.
 * Bull, Rue Jean Jaures, B.P.68, 78340, Les Clayes-sous-Bois
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation
 * version 2.1 of the License.
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA  02110-1301, USA.
 **/

package org.ow2.orchestra.facade.runtime.impl;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;

import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.adapters.XmlJavaTypeAdapter;
import javax.xml.namespace.QName;

import org.ow2.orchestra.facade.runtime.CorrelationSetInitialization;
import org.ow2.orchestra.facade.uuid.ActivityInstanceUUID;
import org.ow2.orchestra.util.JAXBUtils;
import org.ow2.orchestra.util.XmlConstants;
import org.ow2.orchestra.util.XmlUtil;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

/**
 *
 * @author Gael Lalire
 */
@XmlRootElement(name = "correlationSetInitialisation")
public class CorrelationSetInitializationImpl implements CorrelationSetInitialization {

  private static final long serialVersionUID = -2069580133901045271L;

  @XmlElement
  private ActivityInstanceUUID activityUUID;
  @XmlElement
  private Date date;
  @XmlElement
  private String name;
  @XmlElement
  @XmlJavaTypeAdapter(JAXBUtils.JAXBMapNodeAdapter.class)
  protected Map<QName, Node> propertyValues;

  public CorrelationSetInitializationImpl() {}

  public CorrelationSetInitializationImpl(final CorrelationSetInitialization record) {
    this.activityUUID = record.getActivityUUID();
    this.date = new Date(record.getDate().getTime());
    this.name = record.getName();
    this.propertyValues = new HashMap<QName, Node>();
    for (final Entry<QName, Node> entry : record.getPropertyValues().entrySet()) {
      this.propertyValues.put(entry.getKey(), entry.getValue().cloneNode(true));
    }
  }

  public ActivityInstanceUUID getActivityUUID() {
    return this.activityUUID;
  }

  public Date getDate() {
    return new Date(this.date.getTime());
  }

  public String getName() {
    return this.name;
  }

  public Map<QName, Node> getPropertyValues() {
    return this.propertyValues;
  }

  public CorrelationSetInitialization copy() {
    return new CorrelationSetInitializationImpl(this);
  }

  private void writeObject(final java.io.ObjectOutputStream out) throws IOException {
    ObjectOutputStream.PutField putField = out.putFields();
    putField.put("activityUUID", activityUUID);
    putField.put("date", date);
    putField.put("name", name);

    if (propertyValues != null) {
      Map<QName, String> newPropertyValues = new HashMap<QName, String>();
      for (Entry<QName, Node> propertyValue : propertyValues.entrySet()) {
        final Element el = XmlUtil.getDocumentWithOneElement(
        new QName(XmlConstants.ELT_DEFAULT_NS, XmlConstants.ELT_DEFAULT_ELEMENT));
        el.appendChild(el.getOwnerDocument().importNode(propertyValue.getValue(), true));
        newPropertyValues.put(propertyValue.getKey(), XmlUtil.toString(el.getOwnerDocument()));
      }
      putField.put("propertyValues", newPropertyValues);
    }

    out.writeFields();
  }

  private void readObject(final java.io.ObjectInputStream in) throws IOException, ClassNotFoundException {
    ObjectInputStream.GetField getFields = in.readFields();
    this.activityUUID = (ActivityInstanceUUID) getFields.get("activityUUID", null);
    this.date = (Date) getFields.get("date", null);
    this.name = (String) getFields.get("name", null);

    this.propertyValues = new HashMap<QName, Node>();
    Map<QName, String> newPropertyValues = (Map<QName, String>) getFields.get("propertyValues", new HashMap<QName,String>());
    for (Entry<QName, String> newPropertyValue : newPropertyValues.entrySet()) {
      Document doc = XmlUtil.getDocumentFromString(newPropertyValue.getValue());
      Node n = null;
      if (doc != null) {
        final Element elt = doc.getDocumentElement();
        final Element child = XmlUtil.element(elt);
        if (child != null) {
          n = child;
        } else {
          n = elt.getFirstChild();
        }
      }
      this.propertyValues.put(newPropertyValue.getKey(), n);
    }
  }
}
