/**
 * Copyright (C) 2008 Bull S. A. S.
 * Bull, Rue Jean Jaures, B.P.68, 78340, Les Clayes-sous-Bois
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation
 * version 2.1 of the License.
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA  02110-1301, USA.
 **/

package org.ow2.orchestra.facade;

import java.util.Map;
import javax.xml.namespace.QName;
import org.ow2.orchestra.facade.def.ProcessDefinition;
import org.ow2.orchestra.facade.exception.ProcessNotFoundException;
import org.ow2.orchestra.facade.uuid.ProcessDefinitionUUID;

/**
 * Management operations.<br>
 *
 * @author Guillaume Porcher
 */
public interface ManagementAPI {

  /**
   * To deploy a process giving the bytes table of an archive file.<br>
   * The archive should contains the .bpel file and the .wsdl files.
   * @param barFile The bytes table of the bar file.
   * @return The recorded data of the deployed process.
   */
  ProcessDefinition deployBar(byte[] barFile);

  /**
   * To deploy the package giving the deployment object.
   * @param deployment The deployment object that should contain the bpel file and the wsdl definitions.
   * @return The recorded data of the deployed process.
   */
  ProcessDefinition deploy(Deployment deployment);

  /**
   * To undeploy the process giving the qualified name of the process.
   * @param processQName The qualified name.
   * @return true if the process has been succesfully undeployed otherwise
   * returns false.
   */
  boolean undeploy(QName processQName);

  /**
   * To undeploy the process giving the uuid of the process.
   * @param processDefinitionUUID te uuid of the process
   * @return true if the process has been succesfully undeployed otherwise
   * returns false.
   */
  boolean undeploy(ProcessDefinitionUUID processDefinitionUUID);

  /**
   * To delete monitoring information on the process giving the uuid of the process.
   * @param processDefinitionUUID te uuid of the process
   * @return true if the process has been succesfully deleted otherwise
   * returns false.
   */
  boolean remove(ProcessDefinitionUUID processDefinitionUUID);

  /**
   * Redeploy all web services.
   * Use after a reboot to restore the previous environment.
   */
  void initialize();

  /**
   * Deploy web services for the process processDefinitionUUID on this server.
   * @param processDefinitionUUID
   */
  void deployWebServices(final ProcessDefinitionUUID processDefinitionUUID);

  /**
   * Undeploy web services for the process processDefinitionUUID on this server.
   * @param processDefinitionUUID
   */
  void undeployWebServices(final ProcessDefinitionUUID processDefinitionUUID);

  /**
   * Returns URI of the WSDL for a given process.
   * This URI doesn't contain ${orchestra.ws.url}, it's the relative path starting to orchestra servlet.
   * @param processDefinitionUUID process UUID
   * @return A Map containing binding name as key and wsdl location as value
   */
  Map<String, String> getWSContractLocation(final ProcessDefinitionUUID processDefinitionUUID);
  /**
   * Activate the process with the given uuid.
   * All other versions of the process are set to retired state ({@link #retire(ProcessDefinitionUUID)}.
   * New instances of the process will use this version of the process.
   * @param processDefinitionUUID the uuid of the process definition.
   * @return <code>true</code> if the process has been set to active. <code>false</code> if the process was already active.
   * @throws ProcessNotFoundException if processDefinitionUUID cannot be found
   */
  boolean retire(ProcessDefinitionUUID processDefinitionUUID) throws ProcessNotFoundException;

  /**
   * Mark the process with the given UUID as retired.
   * All running instances of the process can continue execution, but no new instances can be created.
   * @param processDefinitionUUID the uuid of the process definition.
   * @return <code>true</code> if the process has been set to retired. <code>false</code> if the process was already retired.
   * @throws ProcessNotFoundException if processDefinitionUUID cannot be found
   */
  boolean activate(ProcessDefinitionUUID processDefinitionUUID) throws ProcessNotFoundException;


  /**
  * Refresh the process state from the database (used to propagate process information in a cluster).
  * @param processDefinitionUUID the uuid of the process definition.
  */
  void refreshProcess(ProcessDefinitionUUID processDefinitionUUID);

  /**
   * @param namespace the namespace of the extension to check.
   * @return true if this extension is supported by this instance of Orchestra.
   */
  boolean isExtensionSupported(String namespace);
}
