/**
 * Copyright (C) 2011  Bull S. A. S.
 * Bull, Rue Jean Jaures, B.P.68, 78340, Les Clayes-sous-Bois
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation
 * version 2.1 of the License.
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA  02110-1301, USA.
 **/
package org.ow2.orchestra.facade.criteria;

import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlElementWrapper;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlTransient;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

/**
 * Criteria class allows to restrict a monitoring request according to criteria.
 * <p>
 * Supported criteria are:
 * <ul>
 * <li>
 * firstResult: the start index for this page. The default value is '0'.</li>
 * <li>
 * maxResults: the maximum number of results to be returned by this page (the
 * actual number of results may be lesser or even could be 0). The default value
 * is '-1' and a negative value is treated as meaning no limit .</li>
 * <li>
 * order: a sort order (ascending or descending) on a given field. The default
 * value is no order.</li>
 * <li>
 * restrictions: a list filters associated by a logical <code>and</code> expression. The default value is no filters.</li>
 * </ul>
 * </p>
 *
 * @author Loic Albertin
 * @see PagedResult
 * @see org.ow2.orchestra.facade.QueryRuntimeAPI
 * @see SortableField
 * @see FilterField
 */
@XmlRootElement(name = "criteria")
public class Criteria<T> implements Serializable {

  /**
   *
   */
  private static final long serialVersionUID = 6733586963303203831L;

  /**
   * negative max result are treated as meaning no limit
   */
  @XmlElement
  private int maxResults = -1;

  /**
   * Default first result is the first one
   */
  @XmlElement
  private int firstResult = 0;

  /**
   * Default order is null
   */
  @XmlElement
  private Order<T> order = null;

  /**
   * Restrictions used on this criteria.
   * An empty list (the default) means no restrictions.
   */
  @XmlElementWrapper
  private List<Restriction<T>> restrictions = new ArrayList<Restriction<T>>();

  public Criteria() {
  }

  /**
   * @return the maxResults
   */
  @XmlTransient
  public final int getMaxResults() {
    return maxResults;
  }

  /**
   * @param maxResults the maxResults to set, a negative value is treated as
   *                   meaning no limit (this is the default behavior)
   * @return the updated criteria
   */
  public final Criteria<T> setMaxResults(final int maxResults) {
    this.maxResults = maxResults;
    return this;
  }

  /**
   * @return the firstResult
   */
  @XmlTransient
  public final int getFirstResult() {
    return firstResult;
  }

  /**
   * @param firstResult the firstResult to set, 0 is the default value
   * @return the updated criteria
   */
  public final Criteria<T> setFirstResult(final int firstResult) {
    this.firstResult = firstResult;
    return this;
  }

  /**
   * @return the order, returns null no order is set (default)
   */
  @XmlTransient
  public final Order<T> getOrder() {
    return order;
  }

  /**
   * @param order the order to set
   * @return the updated criteria
   */
  public final Criteria<T> setOrder(final Order<T> order) {
    this.order = order;
    return this;
  }


  /**
   * @param field a {@link SortableField} on which a ascending sort should be used
   * @return the updated criteria
   */
  public final Criteria<T> setAscendingOrder(final SortableField<T> field) {
    this.order = Order.ascending(field);
    return this;
  }

  /**
   * @param field a {@link SortableField} on which a ascending sort should be used
   * @return the updated criteria
   */
  public final Criteria<T> setDescendingOrder(final SortableField<T> field) {
    this.order = Order.descending(field);
    return this;
  }

  /**
   * Adds a restriction to the criteria
   *
   * @param restriction The {@link Restriction} to add
   * @return the updated criteria
   */
  public final Criteria<T> addRestriction(final Restriction<T> restriction) {
    this.restrictions.add(restriction);
    return this;
  }

  /**
   * @return the List of restrictions, returns an empty list if no restrictions are set (default)
   */
  public final List<Restriction<T>> getRestrictions() {
    return restrictions;
  }
}
