/**
 * Copyright (C) 2007  Bull S. A. S.
 * Bull, Rue Jean Jaures, B.P.68, 78340, Les Clayes-sous-Bois
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation
 * version 2.1 of the License.
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA  02110-1301, USA.
 **/
package org.ow2.orchestra.facade.runtime.impl;

import org.ow2.orchestra.facade.runtime.ActivityState;
import org.ow2.orchestra.facade.runtime.RuntimeRecord;
import org.ow2.orchestra.facade.uuid.ProcessDefinitionUUID;
import org.ow2.orchestra.facade.uuid.ProcessInstanceUUID;
import org.ow2.orchestra.util.Misc;

import javax.xml.bind.annotation.XmlElement;
import java.util.Date;

/**
 *
 * @author Gael Lalire
 */
public abstract class RuntimeRecordImpl implements RuntimeRecord {

  private static final long serialVersionUID = -634690080367047966L;
  @XmlElement
  protected Date startedDate;
  @XmlElement
  protected Date endedDate;
  @XmlElement
  protected ProcessDefinitionUUID processUUID;
  @XmlElement
  protected ProcessInstanceUUID instanceUUID;

  /**
   * The activity state.
   */
  @XmlElement
  protected ActivityState activityState;

  protected RuntimeRecordImpl() {}

  protected RuntimeRecordImpl(final RuntimeRecord src) {
    Misc.checkArgsNotNull(src);
    this.processUUID = src.getProcessDefinitionUUID();
    this.instanceUUID = src.getProcessInstanceUUID();
    this.processUUID = src.getProcessDefinitionUUID();
    this.startedDate = new Date(src.getStartedDate().getTime());
    this.activityState = src.getState();
    if (src.getEndedDate() != null) {
      this.endedDate = src.getEndedDate();
    }
  }

  public ProcessDefinitionUUID getProcessDefinitionUUID() {
    return this.processUUID;
  }

  public ProcessInstanceUUID getProcessInstanceUUID() {
    return this.instanceUUID;
  }

  public Date getEndedDate() {
    if (this.endedDate == null) {
      return null;
    }
    return new Date(this.endedDate.getTime());
  }

  public Date getStartedDate() {
    return new Date(this.startedDate.getTime());
  }

  /**
   * Call this method when the runtime is ended.
   */
  public void end(final ActivityState finishState) {
    this.endedDate = new Date();
    this.activityState = finishState;
  }

  @Override
  public int hashCode() {
    return this.processUUID.hashCode();
  }

  public ActivityState getState() {
    return this.activityState;
  }

  @Override
  public boolean equals(final Object obj) {
    if (obj == null) {
      return false;
    }
    if (!(obj.getClass().equals(this.getClass()))) {
      return false;
    }
    final RuntimeRecordImpl other = (RuntimeRecordImpl) obj;
    if (!this.getState().equals(other.getState())) {
      return false;
    }
    if (!other.processUUID.equals(this.processUUID)) {
      return false;
    }
    if (!other.instanceUUID.equals(this.instanceUUID)) {
      return false;
    }
    return true;
  }
}
