/**
 * Copyright (C) 2007-2009  Bull S. A. S.
 * Bull, Rue Jean Jaures, B.P.68, 78340, Les Clayes-sous-Bois
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation
 * version 2.1 of the License.
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA  02110-1301, USA.
 */
package org.ow2.orchestra.axis;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.wsdl.Binding;
import javax.wsdl.BindingOperation;
import javax.wsdl.Fault;
import javax.wsdl.Operation;
import javax.wsdl.Part;
import javax.wsdl.Port;
import javax.wsdl.Service;
import javax.wsdl.extensions.ExtensibilityElement;
import javax.wsdl.extensions.soap.SOAPAddress;
import javax.wsdl.extensions.soap.SOAPBinding;
import javax.wsdl.extensions.soap.SOAPBody;
import javax.wsdl.extensions.soap.SOAPOperation;
import javax.xml.namespace.QName;
import javax.xml.soap.Detail;
import javax.xml.soap.DetailEntry;
import javax.xml.soap.Name;
import javax.xml.soap.SOAPBodyElement;
import javax.xml.soap.SOAPFault;
import javax.xml.soap.SOAPMessage;

import org.ow2.orchestra.exception.FaultWithMessage;
import org.ow2.orchestra.facade.exception.OrchestraRuntimeException;
import org.ow2.orchestra.pvm.internal.util.XmlUtil;
import org.ow2.orchestra.services.OperationKey;
import org.ow2.orchestra.services.itf.Invoker;
import org.ow2.orchestra.util.SOAPUtil;
import org.ow2.orchestra.var.Message;
import org.w3c.dom.Element;

/**
 * Invoker.java.
 *
 * @author Goulven Le Jeune, Charles Souillard, Candy Chlad, Stanislas Giraudet
 *         De Boudemange, Guillaume Porcher, S. Ali Tokmen
 *
 *         Created on : Jul 3, 2007
 */
public class AxisInvoker implements Invoker {

  private static Logger log = Logger.getLogger(AxisInvoker.class.getName());

  /**
   * perform an invoke using the given parameters.
   *
   * @param instance the instance of the process
   * @param name the name of the invoke activity
   * @param partnerEndPointReference the endpoint of the partner service to
   *          invoke
   * @param operation the operation of the invoke activity
   * @param messageToSend the value of the message used in invoke
   * @return the message received in response of this call (if there is one...)
   */
  public Message invoke(final OperationKey operationKey, final Element partnerEndPointReference, final Message requestMessage,
    final List<Service> services) {

    if (AxisInvoker.log.isLoggable(Level.FINER)) {
      AxisInvoker.log.entering(AxisInvoker.class.getName(), "invoke", new Object[] {
          operationKey, partnerEndPointReference, requestMessage
      });
    }

    try {
      // We find the portType specified for the partner role
      final QName portTypeQName = operationKey.getPortTypeQName();
      final String operationName = operationKey.getOperationName();

      // We take the first binding of the WSDL definition which binds the
      // portType
      //
      // TODO: Time complexity of this algorithm is high! (O(s*p*e)) !
      final List<Binding> bindings = new ArrayList<Binding>();
      for (final Service service : services) {
        for (final Port port : (Collection<Port>) service.getPorts().values()) {
          final List<ExtensibilityElement> elements = port.getExtensibilityElements();
          for (final ExtensibilityElement element : elements) {
            if (!(element instanceof SOAPAddress)) {
              AxisInvoker.log.warning("The extensibility element: " + element.getElementType() + " is not supported!");
            } else {
              if (port.getBinding().getPortType().getQName().equals(portTypeQName)) {
                bindings.add(port.getBinding());
              }
            }
          }
        }
      }
      if (bindings.size() == 0) {
        throw new OrchestraRuntimeException("Can't find any supported binding!");
      }

      final Binding binding = bindings.get(0);
      AxisInvoker.log.fine("Using the first binding supported: " + binding.getQName());

      // Get the Operation associated with the portType.
      final Operation operation = binding.getPortType().getOperation(operationName, null, null);
      // Get the partner endpoint address
      final String endpoint = AxisInvoker.resolveEndPoint(partnerEndPointReference);

      // style used for the operation (RPC or document)
      String style = null;
      // Use used for the input of the operation
      String use = null;
      String soapAction = null;

      // we look for the default style (Document) or (RPC) to use for
      // operations.
      for (final Object elem : binding.getExtensibilityElements()) {
        // QName elementQName = ((ExtensibilityElement) elem).getElementType();
        // if (elementQName.equals(SOAP_BINDING)) {
        if (elem instanceof SOAPBinding) {
          final SOAPBinding soapBinding = (SOAPBinding) elem;
          style = soapBinding.getStyle();
          break;
        }
      }

      // get particular style configuration for operation and URI of soapAction
      final BindingOperation opBinding = binding.getBindingOperation(operationName, null, null);
      for (final Object elem : opBinding.getExtensibilityElements()) {
        // QName elementQName = ((ExtensibilityElement) elem).getElementType();
        // if (elementQName.equals(SOAP_OPERATION)) {
        if (elem instanceof SOAPOperation) {
          final SOAPOperation soapOperation = (SOAPOperation) elem;
          if (soapOperation.getStyle() != null) {
            style = soapOperation.getStyle();
          }
          soapAction = soapOperation.getSoapActionURI();
          break;
        }
      }

      // get the use of the Input (Literal or encoded)
      for (final Object elem : opBinding.getBindingInput().getExtensibilityElements()) {
        // QName elementQName = ((ExtensibilityElement) elem).getElementType();
        // if (elementQName.equals(SOAP_BODY)) {
        if (elem instanceof SOAPBody) {
          final SOAPBody soapBody = (SOAPBody) elem;
          use = soapBody.getUse();
          break;
        }
      }

      if (soapAction == null) {
        throw new OrchestraRuntimeException("No soapAction specified for this operation : " + operationName);
      }

      final Map inputParts = operation.getInput().getMessage().getParts();
      if (style.equals("document") && inputParts.size() != 1) {
        throw new OrchestraRuntimeException("Style is document but input message has many parts : "
          + "not supported as WSI Basic profiles requires only one part for document style");
      }
      if (AxisInvoker.log.isLoggable(Level.FINE)) {
        AxisInvoker.log.fine("Create the call to the endPoint['" + endpoint + "']");
      }

      SOAPMessage request = null;
      if (style.equals("document")) {
        if (requestMessage.getParts().size() != 1) {
          throw new OrchestraRuntimeException("Trying to perform a call with many parts using document style : "
            + "WSI Basic profiles does not allow that : only one part is expected (having the name of the operation");
        }
        request = SOAPUtil.buildDocumentSOAPMessage(soapAction, requestMessage);
      } else {
        request = SOAPUtil.buildRpcSOAPMessage(soapAction, requestMessage, operationName);
      }
      final SOAPMessage response = SOAPUtil.call(request, endpoint);
      if (operation.getOutput() != null) {
        final Map outputParts = operation.getOutput().getMessage().getParts();

        if (style.equals("document") && outputParts.size() != 1) {
          throw new OrchestraRuntimeException("Style is document but output message has many parts : "
            + "not supported as WSI Basic profiles requires only one part for document style");
        }
        if (response == null) {
          throw new OrchestraRuntimeException("This operation(" + binding.getQName() + ":" + operationName
            + ") requires a response but no response was received");
        }

        final javax.xml.soap.SOAPBody soapBody = response.getSOAPBody();
        if (soapBody.hasFault()) {
          Message faultMessage = null;

          final SOAPFault soapFault = response.getSOAPBody().getFault();
          final Name faultName = soapFault.getFaultCodeAsName();
          final QName faultQName = new QName(faultName.getURI(), faultName.getLocalName());
          if (portTypeQName.getNamespaceURI().equals(faultName.getURI())) {
            // This is a WSDL fault, get message
            final Fault fault = operation.getFault(faultName.getLocalName());
            if (fault != null) {
              final Detail detail = soapFault.getDetail();
              if (detail != null) {
                final Iterator<DetailEntry> detailEntries = detail.getDetailEntries();
                if (detailEntries.hasNext()) {
                  final DetailEntry detailEntry = detailEntries.next();
                  // style for faults is always document (cf http://www.w3.org/TR/wsdl#_soap:fault )
                  faultMessage = SOAPUtil.buildMessageFromDocumentSOAPBodyElement(
                      fault.getMessage().getParts().values(), detailEntry);
                }
              }
            }
          }
          throw new FaultWithMessage(faultQName, faultMessage);
        } else {
          return this.getMessageSOAPFromResponse(response, outputParts, style, use);
        }
      } else {
        // operation.getOutput() is null
        return null;
      }
    } catch (final OrchestraRuntimeException oe) {
      throw oe;
    } catch (final Exception e) {
      throw new OrchestraRuntimeException("Exception caught while invoke WS", e);
    }
  }

  private Message getMessageSOAPFromResponse(final SOAPMessage response, final Map<String, Part> outputParts,
    final String style, final String use) {

    try {
      final Iterator<javax.xml.soap.SOAPBodyElement> bodyElementsIterator = response.getSOAPBody().getChildElements();
      if (!bodyElementsIterator.hasNext()) {
        throw new OrchestraRuntimeException("Exception caught while building a message from soapResponse "
          + "(one and only one element was expected as a child of body but 0 were received) : " + response);
      }
      final SOAPBodyElement element = bodyElementsIterator.next();
      if (bodyElementsIterator.hasNext()) {
        throw new OrchestraRuntimeException("Exception caught while building a message from soapResponse "
          + "(only one element was expected as a child of body but many were received) : " + response);
      }
      if (style.equals("document")) {
        return SOAPUtil.buildMessageFromDocumentSOAPBodyElement(outputParts.values(), element);
      } else {
        return SOAPUtil.buildMessageFromRpcSOAPBodyElement(outputParts.values(), element);
      }

    } catch (final Exception e) {
      throw new OrchestraRuntimeException("Exception caught while building a message from soapResponse : " + response, e);
    }
  }

  /**
   * Extract the endPoint reference from a service reference information item.
   *
   * @return the end Point URL
   */
  public static String resolveEndPoint(final Element serviceRefElem) {
    final org.w3c.dom.Element endpointRef = XmlUtil.element(serviceRefElem, "EndpointReference");
    final org.w3c.dom.Element address = XmlUtil.element(endpointRef, "Address");
    return address.getTextContent();
  }

}
