/**
 * Copyright (C) 2006  Bull S. A. S.
 * Bull, Rue Jean Jaures, B.P.68, 78340, Les Clayes-sous-Bois
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation
 * version 2.1 of the License.
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA  02110-1301, USA.
 **/
package org.ow2.orchestra.axis;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.wsdl.Operation;
import javax.wsdl.Output;
import javax.wsdl.Part;
import javax.wsdl.PortType;
import javax.xml.namespace.QName;

import org.ow2.orchestra.definition.BpelProcess;
import org.ow2.orchestra.env.EnvFactoryRepository;
import org.ow2.orchestra.pvm.env.Environment;
import org.ow2.orchestra.pvm.env.EnvironmentFactory;
import org.ow2.orchestra.pvm.internal.cmd.Command;
import org.ow2.orchestra.pvm.internal.cmd.CommandService;
import org.ow2.orchestra.runtime.BpelExecution;
import org.ow2.orchestra.services.MessageCarrier;
import org.ow2.orchestra.services.OperationKey;
import org.ow2.orchestra.services.ReceivingService;
import org.ow2.orchestra.services.itf.Repository;
import org.ow2.orchestra.util.SOAPUtil;
import org.ow2.orchestra.var.Message;
import org.ow2.orchestra.wsdl.WsdlsInfos;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

public final class AxisWSImpl {

  private static Logger log = Logger.getLogger(AxisWSImpl.class.getName());

  private AxisWSImpl() {
  }

  protected static class BuildIncomingMessageCommand implements Command<Message> {

    private final QName processQName;
    private final String operationStyle;
    private final Element[] bodies;
    private final Operation operation;

    public BuildIncomingMessageCommand(final Operation operation, final QName processQName, final String operationStyle,
      final Element[] bodies) {
      this.operation = operation;
      this.processQName = processQName;
      this.operationStyle = operationStyle;
      this.bodies = bodies;
    }

    public Message execute(final Environment environment) throws Exception {
      final Repository repository = environment.get(Repository.class);

      final BpelProcess bpelProcess = repository.getProcess(this.processQName);
      final WsdlsInfos wsdlsInfos = bpelProcess.getWsdlInfos();

      final javax.wsdl.Message inputMessage = wsdlsInfos.getMessage(this.operation.getInput().getMessage().getQName());
      if (this.operationStyle.equals("document")) {
        return SOAPUtil.buildMessageFromDocumentSOAPBodyElement(inputMessage.getParts().values(), this.bodies[0]);
      } else {
        return SOAPUtil.buildMessageFromRpcSOAPBodyElement(inputMessage.getParts().values(), this.bodies[0]);
      }
    }
  }

  protected static class GetOperationCommand implements Command<Operation> {

    private final QName processQName;
    private final QName portTypeQName;
    private final String operationName;

    public GetOperationCommand(final QName processQName, final QName portTypeQName, final String operationName) {
      this.processQName = processQName;
      this.portTypeQName = portTypeQName;
      this.operationName = operationName;
    }

    public Operation execute(final Environment environment) throws Exception {
      final Repository repository = environment.get(Repository.class);

      final BpelProcess bpelProcess = repository.getProcess(this.processQName);
      final WsdlsInfos wsdlsInfos = bpelProcess.getWsdlInfos();

      return AxisWSImpl.getOperation(wsdlsInfos.getPortType(this.portTypeQName), this.operationName);

    }

  }

  public static Element[] call(final Element[] bodies, final QName processQName, final QName portTypeQName,
    final String operationName, final String operationStyle) {

    if (AxisWSImpl.log.isLoggable(Level.FINE)) {
      AxisWSImpl.log.fine("Entering WS Implementation, process = " + processQName + ", PT = " + portTypeQName + ", op = " + operationName
        + ", opStyle = " + operationStyle);
      for (final Element body : bodies) {
        AxisWSImpl.log.fine("body = " + body);
      }
    }
    Element[] elementsReturned = null;
    Message incomingMessage = null;

    final EnvironmentFactory environmentFactory = EnvFactoryRepository.get();
    final CommandService commandService = environmentFactory.get(CommandService.class);

    final Operation operation = commandService.execute(new GetOperationCommand(processQName, portTypeQName, operationName));

    final OperationKey operationKey = new OperationKey(processQName, portTypeQName, operationName);
    incomingMessage = commandService.execute(new BuildIncomingMessageCommand(operation, processQName, operationStyle, bodies));

    if (AxisWSImpl.log.isLoggable(Level.FINE)) {
      AxisWSImpl.log.fine("incomingMessage = " + incomingMessage);
    }

    MessageCarrier messageCarrier = null;
    BpelExecution instance = null;

    if (operation.getOutput() != null) {
      // it is a 2 ways operation. We have to get an identifier.
      messageCarrier = new MessageCarrier();
    }
    if (AxisWSImpl.log.isLoggable(Level.FINE)) {
      AxisWSImpl.log.fine("Doing call to receiver, messageCarrier = " + messageCarrier);
    }
    instance = ReceivingService.handle(incomingMessage, operationKey, messageCarrier);
    if (AxisWSImpl.log.isLoggable(Level.FINE)) {
      AxisWSImpl.log.fine("call to receiver done. Instance = " + instance);
    }
    if (operation.getOutput() != null) {
      if (AxisWSImpl.log.isLoggable(Level.FINE)) {
        AxisWSImpl.log.fine("This operation has an output, waiting for messageCarrier...");
      }
      final Message responseMessage = messageCarrier.getMessage();
      if (operationStyle.equals("document")) {
        if (AxisWSImpl.log.isLoggable(Level.FINE)) {
          AxisWSImpl.log.fine("This operation is using 'document' style.");
        }
        // We make the response");
        final List<Part> parts = AxisWSImpl.getOutputParts(operation);
        elementsReturned = new Element[parts.size()];
        int i = 0;
        for (final Part part : parts) {
          elementsReturned[i] = responseMessage.getPartValue(part.getName());
          i++;
        }
      } else {
        if (AxisWSImpl.log.isLoggable(Level.FINE)) {
          AxisWSImpl.log.fine("This operation is using 'rpc' style.");
        }
        // it is a rpc style!!
        // We make the response");
        final Document rpcDocument = SOAPUtil.buildRpcDocument(responseMessage, operationName);
        elementsReturned = new Element[1];
        elementsReturned[0] = rpcDocument.getDocumentElement();
      }
    }
    if (AxisWSImpl.log.isLoggable(Level.FINE)) {
      AxisWSImpl.log.fine("elementsReturned = " + Arrays.toString(elementsReturned));
    }
    return elementsReturned;
  }

  private static Operation getOperation(final PortType portType, final String operationName) {
    final List<Operation> operations = portType.getOperations();
    for (final Operation operation : operations) {
      if (operation.getName().equals(operationName)) {
        return operation;
      }
    }
    return null;
  }

  private static List<Part> getOutputParts(final Operation operation) {
    final Output output = operation.getOutput();
    if (output != null) {
      return AxisWSImpl.getMessageParts(output.getMessage());
    }
    return new ArrayList<Part>();
  }

  @SuppressWarnings("unchecked")
  private static List<Part> getMessageParts(final javax.wsdl.Message message) {
    final Map< ? , Part> partMap = message.getParts();
    final Collection<Part> parts = partMap.values();
    return new ArrayList<Part>(parts);
  }
}
