package org.ow2.orchestra.axis;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLConnection;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.wsdl.Binding;
import javax.wsdl.Operation;
import javax.wsdl.OperationType;
import javax.wsdl.Port;
import javax.wsdl.Service;
import javax.wsdl.extensions.ExtensibilityElement;
import javax.wsdl.extensions.soap.SOAPBinding;
import javax.wsdl.extensions.soap.SOAPBody;
import javax.wsdl.extensions.soap.SOAPFault;
import javax.wsdl.extensions.soap12.SOAP12Binding;
import javax.wsdl.extensions.soap12.SOAP12Body;
import javax.wsdl.extensions.soap12.SOAP12Fault;
import javax.xml.namespace.QName;

import org.apache.axis.deployment.wsdd.WSDDConstants;
import org.apache.axis.deployment.wsdd.WSDDProvider;
import org.apache.axis.tools.ant.axis.AdminClientTask;
import org.apache.axis.utils.ClassUtils;
import org.apache.axis.wsdl.toJava.Utils;
import org.apache.tools.ant.DefaultLogger;
import org.apache.tools.ant.Project;
import org.apache.tools.ant.taskdefs.Copy;
import org.apache.tools.ant.taskdefs.Delete;
import org.apache.tools.ant.taskdefs.Expand;
import org.apache.tools.ant.taskdefs.Jar;
import org.apache.tools.ant.taskdefs.Javac;
import org.apache.tools.ant.taskdefs.Property;
import org.apache.tools.ant.types.FileSet;
import org.apache.tools.ant.types.Path;
import org.ow2.orchestra.definition.BpelProcess;
import org.ow2.orchestra.deployment.Deployment;
import org.ow2.orchestra.exception.OrchestraException;
import org.ow2.orchestra.jmx.JMXAgent;
import org.ow2.orchestra.services.MutexRepository;
import org.ow2.orchestra.services.MutexRepository.ProcessLock;
import org.ow2.orchestra.ws.WSDeployer;

/**
 * Axis engine WS deployer.
 */
public class AxisDeployer extends WSDeployer {

  // WSDD constants
  /** deploy.wsdd. */
  public static final String WSDD_DEPLOY_FILE_NAME = "deploy.wsdd";

  /** undeploy.wsdd. */
  public static final String WSDD_UNDEPLOY_FILE_NAME = "undeploy.wsdd";

  /** axis deployment namespace. */
  public static final String WSDD_NS = "http://xml.apache.org/axis/wsdd/";

  /** wsdd root element name. */
  public static final String WSDD_SERVICE = "service";

  /** wsdd root element style attribute name. */
  public static final String WSDD_SERVICE_STYLE_ATTR = "style";

  /** wsdd root element provider attribute name. */
  public static final String WSDD_SERVICE_PROVIDER_ATTR = "provider";

  /** wsdd style attribute message value. */
  public static final String MESSAGE_STYLE = "Message";

  /** utility var to rename wsdd file with a suffix. */
  public static final String NEW_WSDD_SUFFIX = ".message";

  /** scope of the wsdd generated. */
  public static final String SCOPE = "Request";

  /** jdkVersion used to compile generated java files. */
  public static final String JDK_VERSION = "1.5";

  /** utility variable to define ant environment. */
  public static final String ANT_ENV = "antEnv";

  /** log. */
  private static Logger log = Logger.getLogger(Deployment.class.getName());

  private Project antProject;

  private File tmpDir;

  private AxisConfiguration axisConfiguration;

  private BpelProcess bpelProcess;

  /**
   * Default constructor.
   *
   * @param wsId -
   *            id of the WS : in our case, processName
   * @param wsdlUrl -
   *            url of the wsdl file to deploy.
   * @param orchestraDirectory -
   *            Absolute path to the orchestra directory.
   */
  /*
   * public AxisDeployer(final BpelProcess bpelProcess, URL wsdlURL) {
   * super(bpelProcess.getQName()); this.bpelProcess = bpelProcess; this.wsdlURL =
   * wsdlURL; //this.wsdlDefinition = WsdlUtil.readWsdl(wsdlURL);
   * setAntProject(); this.axisConfiguration = AxisConfiguration.getInstance();
   * setProxy(); }
   */

  public AxisDeployer(final BpelProcess bpelProcess) {
    super(bpelProcess.getQName());
    this.bpelProcess = bpelProcess;
    setAntProject();
    this.axisConfiguration = AxisConfiguration.getInstance();
    setProxy();
  }

  /**
   * return null if operation style is supported by Axis engine. Else, returns a
   * message explaining the problem.
   *
   * @param operationStyle -
   *            operationStyle to check
   * @return null if operation style is supported by Axis engine. Else, returns
   *         a message explaining the problem.
   */
  protected String checkOperationStyle(final String operationStyle) {
    if (operationStyle == null) {
      return "Style attribute of this operation must be specified";
    } else if (!operationStyle.equals("document")
        && !operationStyle.equals("rpc")) {
      return "Style attribute of this operation must be : document or rpc";
    }
    return null;
  }

  /**
   * return null if operation type is supported by Axis engine. Else, returns a
   * message explaining the problem.
   *
   * @param operationType -
   *            operationType to check
   * @return null if operation type is supported by Axis engine. Else, returns a
   *         message explaining the problem.
   */
  protected String checkOperationType(final OperationType operationType) {
    if (!operationType.equals(OperationType.REQUEST_RESPONSE)
        && !operationType.equals(OperationType.ONE_WAY)) {
      return "Operation type : " + operationType
          + " is not supported. Please use one of : " + OperationType.ONE_WAY
          + "/" + OperationType.REQUEST_RESPONSE;
    }
    return null;
  }

  /**
   * return null if soapVersion is supported by Axis engine. Else, returns a
   * message explaining the problem.
   *
   * @param soapBinding -
   *            soapBinding to check
   * @return null if soapVersion is supported by Axis engine. Else, returns a
   *         message explaining the problem.
   */
  protected String checkSoapVersion(final ExtensibilityElement soapBinding) {
    String soapVersion = null;
    if (!(soapBinding instanceof SOAPBinding)
        && !(soapBinding instanceof SOAP12Binding)) {
      return "Supported Soap Version are " + URI_WSDL11_SOAP11 + "/"
          + URI_WSDL11_SOAP12;
    }
    return null;
  }

  /**
   * return null if transport is supported by Axis engine. Else, returns a
   * message explaining the problem.
   *
   * @param soapBinding -
   *            soapBinding to check
   * @return null if transport is supported by Axis engine. Else, returns a
   *         message explaining the problem.
   */
  protected String checkTransport(final ExtensibilityElement soapBinding) {
    String transportUri = "";
    if (soapBinding instanceof SOAPBinding) {
      transportUri = ((SOAPBinding) soapBinding).getTransportURI();
    } else if (soapBinding instanceof SOAP12Binding) {
      transportUri = ((SOAP12Binding) soapBinding).getTransportURI();
    }
    if (!SOAP_HTTP_TRANSPORT_URI.equals(transportUri)) {
      return "Transport URI : " + transportUri
          + " is not supported. Please use " + SOAP_HTTP_TRANSPORT_URI;
    }
    return null;
  }

  /**
   * return null if use is supported by Axis engine. Else, returns a message
   * explaining the problem.
   *
   * @param soapBody -
   *            soapBody to check
   * @return null if use is supported by Axis engine. Else, returns a message
   *         explaining the problem.
   */
  protected String checkUse(final ExtensibilityElement element) {
    String use = "";
    if (element instanceof SOAPBody) {
      use = ((SOAPBody) element).getUse();
    } else if (element instanceof SOAP12Body) {
      use = ((SOAP12Body) element).getUse();
    } else if (element instanceof SOAPFault) {
      use = ((SOAPFault) element).getUse();
    } else if (element instanceof SOAP12Fault) {
      use = ((SOAP12Fault) element).getUse();
    }
    if (!"literal".equals(use)) {
      return "Use : " + use + " is not supported. Please use " + "literal";
    }
    return null;
  }

  /**
   * return null if soapBody attributes are supported by Axis engine. Else,
   * returns a message explaining the problem.
   *
   * @param soapBody -
   *            soapBody to check
   * @return null if soapBody attributes are supported supported by Axis engine.
   *         Else, returns a message explaining the problem.
   */
  protected String checkSoapBody(final ExtensibilityElement soapBody) {
    List parts = null;
    if (soapBody instanceof SOAPBody) {
      parts = ((SOAPBody) soapBody).getParts();
    } else if (soapBody instanceof SOAP12Body) {
      parts = ((SOAP12Body) soapBody).getParts();
    }
    if (parts != null) {
      return "SoapBody is using parts attribute which is not currently supported.";
    }
    return null;
  }

  protected String checkSoapFault(final ExtensibilityElement soapFault) {
    String name = null;
    if (soapFault instanceof SOAPFault) {
      name = ((SOAPFault) soapFault).getName();
    } else if (soapFault instanceof SOAP12Fault) {
      name = ((SOAP12Fault) soapFault).getName();
    }
    if (name == null) {
      return "SoapFault is not specifying fault name which is not currently supported.";
    }
    return null;
  }

  /**
   * Check if WS engine is available. Else throw an exception.
   */
  protected void checkWSEngineIsAvailable() {
    String urlString = "http://" + this.axisConfiguration.getHost() + ":"
        + this.axisConfiguration.getPort() + "/"
        + this.axisConfiguration.getWebappName() + "/"
        + this.axisConfiguration.getServletPath() + "/"
        + this.axisConfiguration.getGetVersionSvc();
    try {
      connectURL(urlString);
    } catch (Exception e) {
      throw new OrchestraException(
          "Axis web container is not started or Axis engine is not deployed", e);
    }

  }

  /**
   * This method will deploy the specified WS Service on the choosen ws engine.
   * This method must be overriden by each ws engine Deployer.
   *
   * @param def -
   *            Definition object that represents a WS Service to deploy.
   */
  protected void deployServices(List<Service> services) {
    QName javaURI = new QName(WSDDConstants.URI_WSDD_JAVA,
        WSDDBPELMsgProvider.PROVIDER_NAME);
    WSDDProvider.registerProvider(javaURI, new WSDDBPELMsgProvider());
    ProcessLock processLock = MutexRepository.lockProcess(getProcessQName());
    try {
      createTmpDir();
      try {
        File wsddFile = createWsddDeployFile(services);
        createWsddUndeployFile(services);

        // update implementations
        updateImplementationFiles(services);
        // compile java classes, create a jar and deploy it into WEB-INF/lib
        deployJarToWebServer();

        // load wsdd file into axis webapp
        callAdminClientTask(wsddFile);
      } catch (Exception e) {
        throw new OrchestraException("Exception caught in " + this.getClass(),
            e);
      } finally {
        deleteTmpDir();
      }
    } finally {
      MutexRepository.unlockProcess(processLock);
    }
  }

  /**
   * This method will undeploy the specified WS Service from the choosen ws
   * engine. This method must be overriden by each ws engine Deployer.
   *
   * @param def -
   *            Definition object that represents a WS Service to undeploy.
   */
  protected void undeployServices(List<Service> services) {
    ProcessLock processLock = MutexRepository.lockProcess(getProcessQName());
    try {
      createTmpDir();
      try {
        // extract jar file to get undeploy service descriptor
        extractJarFile();
        for (File undeployWsdd : getUndeployFiles()) {
          callAdminClientTask(undeployWsdd);
        }
      } catch (Exception e) {
        throw new OrchestraException("Exception caught in " + this.getClass(),
            e);
      } finally {
        deleteTmpDir();
      }
    } finally {
      MutexRepository.unlockProcess(processLock);
    }
  }

  public void redeployServices(List<Service> services) {
    ProcessLock processLock = MutexRepository.lockProcess(getProcessQName());
    try {
      createTmpDir();
      try {
        // extract jar file to get undeploy service descriptor
        extractJarFile();
        ClassLoader parent = AxisDeployer.class.getClassLoader();
        URL url = null;
        try {
          url = new File(getJarRepository() + File.separator + getJarName())
              .toURI().toURL();
        } catch (MalformedURLException e) {
          throw new OrchestraException(e);
        }
        URL[] urls = new URL[] { url };
        ProcessClassLoader cl = new ProcessClassLoader(getProcessQName(), urls,
            parent);
        ClassLoaderMap.addCL(getProcessQName(), cl);
        for (File deployWsdd : getDeployFiles()) {
          callAdminClientTask(deployWsdd);
        }
      } catch (Exception e) {
        throw new OrchestraException("Exception caught in " + this.getClass(),
            e);
      } finally {
        deleteTmpDir();
      }
    } finally {
      MutexRepository.unlockProcess(processLock);
    }
  }

  /**
   * Extracts a jar deployed in web server to baseDir.
   */
  private void extractJarFile() {
    String jarRepository = getJarRepository();
    File jarFile = new File(jarRepository + File.separator + getJarName());
    Expand unjar = new Expand();
    unjar.setProject(this.antProject);
    unjar.setSrc(jarFile);
    unjar.setDest(this.tmpDir);
    unjar.setTaskName("unjar");
    unjar.execute();
  }

  /**
   * Call axis admin client with the given wsdd file to deploy/undeploy
   * services.
   *
   * @param wsddFile -
   *            wsdd file to use
   */
  private void callAdminClientTask(final File wsddFile) {
    ClassLoader old = ClassUtils.getDefaultClassLoader();
    try {
      ClassLoader processClassLoader = ClassLoaderMap.getCL(this.processQName);
      ClassUtils.setDefaultClassLoader(processClassLoader);
      AdminClientTask adminClientTask = new AdminClientTask();
      adminClientTask.setProject(this.antProject);
      adminClientTask.init();
      adminClientTask.setUrl("-lhttp://" + this.axisConfiguration.getHost()
          + ":" + this.axisConfiguration.getPort() + "/"
          + this.axisConfiguration.getWebappName() + "/servlet/AxisServlet");
      adminClientTask.setXmlFile(wsddFile);
      adminClientTask.setTaskName("adminClientTask");
      adminClientTask.execute();
    } finally {
      ClassUtils.setDefaultClassLoader(old);
    }
  }

  /**
   * Deploy axis WS jar to web server. For that : - compile wsdl2java + new
   * binding classes - create the jar file - deploy it in web server
   *
   */
  private void deployJarToWebServer() {
    compileClasses();
    File jarFile = createJar();
    File destJarFile = deployJar(jarFile);
    ClassLoader parent = AxisDeployer.class.getClassLoader();
    URL url = null;
    try {
      url = destJarFile.toURI().toURL();
    } catch (MalformedURLException e) {
      throw new OrchestraException(e);
    }
    URL[] urls = new URL[] { url };
    ProcessClassLoader cl = new ProcessClassLoader(getProcessQName(), urls,
        parent);
    ClassLoaderMap.addCL(getProcessQName(), cl);
  }

  /**
   * Creates the WS jar file based on generated classes.
   *
   * @return the created jar file
   */
  private File createJar() {
    File jarFile = getWSJarFile();
    Jar j = new Jar();
    j.setTaskName("jar");
    j.setProject(this.antProject);
    j.setDestFile(jarFile);
    FileSet fs1 = new FileSet();
    fs1.setDir(this.tmpDir);
    j.addFileset(fs1);
    j.execute();
    return jarFile;
  }

  /**
   * Deploys the given jar file to web server.
   *
   * @param jarFile
   *            the jar file
   *
   */
  private File deployJar(final File jarFile) {
    String jarRepository = getJarRepository();
    File destJarFile = new File(jarRepository + File.separator + getJarName());
    Copy c = new Copy();
    c.setProject(this.antProject);
    c.setFile(jarFile);
    c.setTofile(destJarFile);
    c.setTaskName("copy");
    c.execute();
    return destJarFile;
  }

  /**
   * Deploys the given wsdl file to web server.
   *
   * @param jarFile
   *            the jar file
   */
  /*
   * private void storeWsdls() { if (wsdlURL != null) { String wsdlRepository =
   * getWsdlRepository(); File destWsdlFile = new File( wsdlRepository +
   * File.separator + getWsdlName()); OutputStream outStream = null; InputStream
   * inStream = null; try { destWsdlFile.createNewFile(); outStream = new
   * FileOutputStream(destWsdlFile);
   *
   * inStream = wsdlURL.openStream(); int c; while ((c = inStream.read()) != -1) {
   * outStream.write(c); } } catch (Exception e) { throw new
   * OrchestraException(e); } finally { try { if (inStream != null) {
   * inStream.close(); } if (outStream != null) { outStream.close(); } } catch
   * (Exception e) { // nothing: handle exception } } } }
   */
  /**
   * Compiles java classes from basedir.
   */
  private void compileClasses() {
    Javac j = new Javac();
    j.setProject(this.antProject);
    j.setDestdir(this.tmpDir);
    Path srcDir = new Path(this.antProject);
    srcDir.setLocation(this.tmpDir);
    j.setSrcdir(srcDir);

    FileSet fileset = new FileSet();
    fileset.setDir(new File(getWebappLibDir()));
    fileset.setIncludes("**/*.jar");

    Path classpath = new Path(this.antProject);
    classpath.addFileset(fileset);

    j.setClasspath(classpath);

    j.setVerbose(false);
    j.setFork(true);
    j.setIncludes("**/*.java");
    j.setTaskName("javac");
    j.setSource(JDK_VERSION);
    j.setDebug(true);
    j.init();
    j.execute();
  }

  /**
   * Create new java implementations for each binding of the given definition.
   *
   * @param def -
   *            the definition to analyse.
   * @return a list of created classes
   */
  private List<String> updateImplementationFiles(List<Service> services) {
    List<String> writtenClasses = new ArrayList<String>();
    for (Service service : services) {
      for (Port port : (Collection<Port>) service.getPorts().values()) {
        Binding binding = port.getBinding();
        writtenClasses.add(new BindingFileWriter(this.tmpDir, binding,
            getProcessQName()).write());
      }
    }
    return writtenClasses;
  }

  /**
   * Updates the given wsddFile to make it using message style service.
   *
   * @param wsddFile :
   *            wsddFile to update
   * @return the updated wsddFile
   */
  /*
   * private File updateWsddFile(final File wsddFile) { //parse wsdd file
   * Document wsddDoc = BpelUtil.getDocumentFromFile(wsddFile);
   *
   * //for each service, set style to Message Element deploymentElement =
   * wsddDoc.getDocumentElement(); NodeList services =
   * deploymentElement.getElementsByTagName(WSDD_SERVICE); for (int i = 0; i <
   * services.getLength(); i++) { Element service = (Element) services.item(i);
   *
   * Element processNameParameter =
   * service.getOwnerDocument().createElement("parameter");
   * processNameParameter.setAttribute("name", "bpelProcessName");
   * processNameParameter.setAttribute("value",
   * getProcessQName().getLocalPart());
   *
   * Element processNamespaceParameter =
   * service.getOwnerDocument().createElement("parameter");
   * processNamespaceParameter.setAttribute("name", "bpelProcessNamespace");
   * processNamespaceParameter.setAttribute("value",
   * getProcessQName().getNamespaceURI());
   *
   * Element wsdlFile = service.getOwnerDocument().createElement("wsdlFile");
   * wsdlFile.setTextContent(getWsdlRepository() + File.separator +
   * getWsdlName());
   *
   * Attr providerAttribute =
   * service.getOwnerDocument().createAttribute(WSDD_SERVICE_PROVIDER_ATTR);
   * providerAttribute.setValue("java:BPELMsg");
   *
   * service.appendChild(processNameParameter);
   * service.appendChild(processNamespaceParameter);
   * service.appendChild(wsdlFile); service.setAttributeNode(providerAttribute);
   *
   * service.setAttribute(WSDD_SERVICE_STYLE_ATTR, MESSAGE_STYLE); } File
   * newWsddFile = getWsddCopy(wsddFile); BpelUtil.writeXmlFile(wsddDoc,
   * newWsddFile); return newWsddFile; }
   */

  /**
   * Return a copy of the given wsdd file.
   *
   * @param originalWsddFile :
   *            original wsdd file
   * @return the copied wsdd file (the new created one)
   */
  private File getWsddCopy(final File originalWsddFile) {
    return new File(originalWsddFile.getAbsolutePath() + NEW_WSDD_SUFFIX);
  }

  /**
   * Return the deploy wsdd file of the given definition.
   *
   * @param def :
   *            def to analyse
   * @return the deploy wsdd file used by the given def
   */
  private File getDeployWsddFile(String wsdlTargetnamespace) {
    return new File(getWsddFilesDir(wsdlTargetnamespace) + File.separator
        + WSDD_DEPLOY_FILE_NAME);
  }

  /**
   * Return the undeploy wsdd file of the given definition.
   *
   * @param def :
   *            def to analyse
   * @return the undeploy wsdd file used by the given def
   */
  private List<File> getUndeployFiles() {
    List<File> files = new ArrayList<File>();
    fillUndeployFiles(files, this.tmpDir);
    return files;
  }

  private void fillUndeployFiles(List<File> files, File currentDir) {
    for (File f : currentDir.listFiles()) {
      if (f.getName().equals("undeploy.wsdd")) {
        files.add(f);
      } else if (f.isDirectory()) {
        fillUndeployFiles(files, f);
      }
    }
  }

  private List<File> getDeployFiles() {
    List<File> files = new ArrayList<File>();
    fillDeployFiles(files, this.tmpDir);
    return files;
  }

  private void fillDeployFiles(List<File> files, File currentDir) {
    for (File f : currentDir.listFiles()) {
      if (f.getName().equals("deploy.wsdd")) {
        files.add(f);
      } else if (f.isDirectory()) {
        fillDeployFiles(files, f);
      }
    }
  }

  /**
   * Returns the directory where wsdd files of this definition could be found.
   *
   * @param def :
   *            def to analyse
   * @return the directory where wsdd files of this definition could be found
   */
  private String getWsddFilesDir(String wsdlTargetnamespace) {
    String wsddFilesDir = this.tmpDir + File.separator
        + getDirectoryFromPackage(getPackageFromNamespace(wsdlTargetnamespace));
    return wsddFilesDir;
  }

  /*
   * UTILITY METHODS
   */

  /**
   * Set basedir field.
   */
  private void createTmpDir() {
    try {
      new File(System.getProperty("java.io.tmpdir")).mkdirs();
      this.tmpDir = File.createTempFile("orch", null, null);
      tmpDir.delete();
      if (!tmpDir.mkdirs()) {
        throw new IOException("Cannot create the temporary directory '"
            + tmpDir + "'.");
      }
      if (log.isLoggable(Level.FINE)) {
        log.fine("tmpDir created : " + this.tmpDir.getAbsolutePath());
      }
    } catch (Exception e) {
      throw new OrchestraException("Error creating " + this.tmpDir, e);
    }
  }

  /**
   * Returns the jar file of the current ws deployed in web server.
   *
   * @return the jar file of the current ws deployed in web server.
   */
  private String getJarRepository() {
    File repo = new File(getWebappLibDir() + File.separator + "WS_Repo");
    repo.mkdirs();
    return repo.getAbsolutePath();
  }

  /**
   * Returns the wsdl file of the current ws deployed in web server.
   *
   * @return the wsdl file of the current ws deployed in web server.
   */
  private String getWsdlRepository() {
    File repo = new File(getJarRepository() + File.separator + "wsdl");
    repo.mkdirs();
    return repo.getAbsolutePath();
  }

  private String getWebappLibDir() {
    URL envUrl = Thread.currentThread().getContextClassLoader().getResource(JMXAgent.defaultEnvConfigFile);
    String envPath = envUrl.getPath();
    String rootDir = envPath.substring(0, envPath.lastIndexOf(JMXAgent.defaultEnvConfigFile));
    String path = rootDir + ".." + File.separator + "lib";
    if (!new File(path).isDirectory()) {
      throw new OrchestraException("Error while deploying web services : dir " + path + " does not exist.");
    }
    return path;
  }

  /**
   * Returns the jar file of the current ws : this jar is in basedir (not
   * already deployed).
   *
   * @return the jar file of the current ws : this jar is in basedir (not
   *         already deployed).
   */
  private File getWSJarFile() {
    return new File(this.tmpDir + File.separator + getJarName());
  }

  private String getJarName() {
    String packag = Utils.makePackageName(getProcessQName().getNamespaceURI());
    return packag + "__" + getProcessQName().getLocalPart() + ".jar";
  }

  private String getWsdlName() {
    String packag = Utils.makePackageName(getProcessQName().getNamespaceURI());
    return packag + "__" + getProcessQName().getLocalPart() + ".wsdl";
  }

  /**
   * Create a connection to the given url.
   *
   * @param urlString :
   *            url to connect
   * @return the created connection
   */
  private URLConnection connectURL(final String urlString) {
    URLConnection connection = null;
    try {
      URL getVersionURL = new URL(urlString);
      connection = getVersionURL.openConnection();
      connection.connect();
    } catch (Exception e) {
      throw new OrchestraException("Unable to get a connection on URL : ", e);
    }
    return connection;
  }

  /**
   * Deletes the given directory.
   *
   * @param dir -
   *            dir to delete.
   */
  private void deleteDir(final File dir) {
    Delete d = new Delete();
    d.setTaskName("delete");
    d.setDir(dir);
    d.setProject(this.antProject);
    d.execute();
  }

  /**
   * Deletes all generated files of this ws.
   */
  private void deleteTmpDir() {
    try {
      deleteDir(this.tmpDir);
    } catch (Exception e) {
      throw new OrchestraException(this.getClass()
          + ".deleteGeneratedFiles, unable to delete directory : "
          + this.tmpDir, e);
    }
  }

  /**
   * Set the used ant project.
   *
   */
  private void setAntProject() {
    this.antProject = new Project();

    // Class to Write build events to a PrintStream
    DefaultLogger dl = new DefaultLogger();
    dl.setMessageOutputLevel(Project.MSG_INFO);
    dl.setErrorPrintStream(System.out);
    dl.setOutputPrintStream(System.out);
    dl.setEmacsMode(false);
    this.antProject.addBuildListener(dl);

    // Initialise the project. This involves setting the default
    // task definitions and loading the system properties.
    this.antProject.init();

    Property p = new Property();
    p.setEnvironment(ANT_ENV);
    p.setProject(this.antProject);
    p.execute();

  }

  /**
   * Returns the java package that maps to the given namespace.
   *
   * @param ns -
   *            ns
   * @return the java package that maps to the given namespace.
   */
  public static String getPackageFromNamespace(final String ns) {
    String packag = Utils.makePackageName(ns);
    return packag;
  }

  /**
   * Returns the directory structure corresponding to the given package.
   *
   * @param packag -
   *            packag
   * @return the directory structure corresponding to the given package.
   */
  public static String getDirectoryFromPackage(final String packag) {
    String dir = packag.replace('.', File.separatorChar);
    return dir;
  }

  /**
   * Set the proxy (if needed) from the Axis configuration.
   */
  private void setProxy() {
    System.getProperties().put("http.proxyHost",
        this.axisConfiguration.getHttpProxyHost());
    System.getProperties().put("http.proxyPort",
        this.axisConfiguration.getHttpProxyPort());
    System.getProperties().put("http.nonProxyHosts",
        this.axisConfiguration.getHttpNonProxyHosts());
  }

  private File createWsddDeployFile(List<Service> services) {
    File wsddFile = getDeployWsddFile(bpelProcess.getTargetNamespace());
    File wsddDir = wsddFile.getParentFile();
    wsddDir.mkdirs();
    FileWriter wsddFileWriter = null;
    try {
      wsddFile.createNewFile();
      wsddFileWriter = new FileWriter(wsddFile);

      wsddFileWriter.write("<?xml version='1.0' encoding='UTF-8'?>\n");
      wsddFileWriter
          .write("<deployment xmlns='http://xml.apache.org/axis/wsdd/' "
              + "    xmlns:java='http://xml.apache.org/axis/wsdd/providers/java'>\n");

      for (Service service : services) {
        for (Port port : (Collection<Port>) service.getPorts().values()) {
          QName bindingQN = port.getBinding().getQName();

          String className = bindingQN.getLocalPart() + "Impl";
          String firstLetter = className.substring(0, 1);
          String nextLetters = className.substring(1);
          className = firstLetter.toUpperCase() + nextLetters;
          wsddFileWriter.write("<service name='" + port.getName()
              + "' provider='java:BPELMsg' style='Message' use='literal'>\n"
              + "<parameter name='wsdlTargetNamespace' value='"
              + service.getQName().getNamespaceURI() + "'/>\n"
              + "<parameter name='wsdlServiceElement' value='"
              + service.getQName().getLocalPart() + "'/>\n"
              + "<parameter name='wsdlServicePort' value='" + port.getName()
              + "'/>\n" + "<parameter name='className' value='"
              + getPackageFromNamespace(bindingQN.getNamespaceURI()) + "."
              + className + "'/>\n" + "<parameter name='wsdlPortType' value='"
              + port.getBinding().getPortType().getQName().getLocalPart()
              + "'/>\n"
              + "<parameter name='typeMappingVersion' value='1.3'/>\n"
              + "<parameter name='scope' value='Request'/>\n"
              + "<parameter name='bpelProcessName' value='"
              + bpelProcess.getName() + "'/>\n"
              + "<parameter name='bpelProcessNamespace' value='"
              + bpelProcess.getTargetNamespace() + "'/>\n");
          for (Operation operation : (Collection<Operation>) port.getBinding()
              .getPortType().getOperations()) {
            wsddFileWriter.write("<operation name='" + operation.getName() + "' qname='operNs:" + operation.getName() 
                + "' xmlns:operNs='" + port.getBinding().getPortType().getQName().getNamespaceURI() + "' ");
            if (operation.getStyle().equals(OperationType.ONE_WAY)) {
              wsddFileWriter.write("mep='oneway' />\n");
            } else {
              //TODO: add operation parameters and return type description
              wsddFileWriter.write(" />\n");
              
            }
          }
          wsddFileWriter.write("</service>\n");
        }
      }
      wsddFileWriter.write("</deployment>");
    } catch (IOException e) {
      throw new OrchestraException("Error while writting deploy.wsdd file: "
          + e.getMessage(), e);
    } finally {
      if (wsddFileWriter != null) {
        try {
          wsddFileWriter.close();
        } catch (IOException e) {
          // TODO
        }
      }
    }
    return wsddFile;
  }

  private void createWsddUndeployFile(List<Service> services) {
    File wsddFile = new File(
        getDeployWsddFile(bpelProcess.getTargetNamespace()).getParent()
            + File.separator + "undeploy.wsdd");
    FileWriter wsddFileWriter = null;
    try {
      wsddFile.createNewFile();
      wsddFileWriter = new FileWriter(wsddFile);

      wsddFileWriter.write("<?xml version='1.0' encoding='UTF-8'?>\n");
      wsddFileWriter
          .write("<undeployment xmlns='http://xml.apache.org/axis/wsdd/'>\n");

      for (Service service : services) {
        for (Port port : (Collection<Port>) service.getPorts().values()) {
          wsddFileWriter.write("<service name='" + port.getName() + "'/>\n");
        }
      }
      wsddFileWriter.write("</undeployment>");
    } catch (IOException e) {
      throw new OrchestraException("Error while writting undeploy.wsdd file: "
          + e.getMessage(), e);
    } finally {
      if (wsddFileWriter != null) {
        try {
          wsddFileWriter.close();
        } catch (IOException e) {
          // TODO
        }
      }
    }
  }
}
