/**
 * Copyright (C) 2006  Bull S. A. S.
 * Bull, Rue Jean Jaures, B.P.68, 78340, Les Clayes-sous-Bois
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation
 * version 2.1 of the License.
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA  02110-1301, USA.
 **/
package org.ow2.orchestra.axis;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.wsdl.Operation;
import javax.wsdl.Output;
import javax.wsdl.Part;
import javax.wsdl.PortType;
import javax.xml.namespace.QName;

import org.jbpm.pvm.env.Environment;
import org.jbpm.pvm.env.EnvironmentFactory;
import org.jbpm.pvm.internal.cmd.Command;
import org.jbpm.pvm.internal.cmd.CommandService;
import org.ow2.orchestra.definition.BpelProcess;
import org.ow2.orchestra.definition.activity.AbstractActivity;
import org.ow2.orchestra.env.EnvFactoryRepository;
import org.ow2.orchestra.runtime.BpelExecution;
import org.ow2.orchestra.services.MessageCarrier;
import org.ow2.orchestra.services.MutexRepository;
import org.ow2.orchestra.services.OperationKey;
import org.ow2.orchestra.services.Receiver;
import org.ow2.orchestra.services.MutexRepository.InstanceLock;
import org.ow2.orchestra.services.itf.Repository;
import org.ow2.orchestra.util.SOAPUtil;
import org.ow2.orchestra.var.Message;
import org.ow2.orchestra.wsdl.WsdlsInfos;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

public final class AxisWSImpl {

  private static Logger log = Logger
      .getLogger(AbstractActivity.class.getName());

  private AxisWSImpl() {
  }

  protected static class RemoveInstanceCommand implements Command<Object> {

    private BpelExecution instance;

    public RemoveInstanceCommand(final BpelExecution instance) {
      this.instance = instance;
    }

    public Object execute(final Environment environment) throws Exception {
      instance = environment.get(Repository.class).getUpdatedExecution(instance);
      if (instance != null && instance.isEnded()) {
        environment.get(Repository.class).removeInstance(instance);
      }
      return null;
    };
  }


  protected static class BuildIncomingMessageCommand implements Command<Message> {
    private final QName processQName;
    private final String operationStyle;
    private final Element[] bodies;
    private final Operation operation;

    public BuildIncomingMessageCommand(final Operation operation, final QName processQName, final String operationStyle, final Element[] bodies) {
      this.operation = operation;
      this.processQName = processQName;
      this.operationStyle = operationStyle;
      this.bodies = bodies;
    }

    public Message execute(final Environment environment) throws Exception {
      Repository repository = environment.get(Repository.class);

      BpelProcess bpelProcess = repository.getProcess(processQName);
      WsdlsInfos wsdlsInfos = bpelProcess.getWsdlInfos();

      javax.wsdl.Message inputMessage = wsdlsInfos.getMessage(
          operation.getInput().getMessage().getQName());
      if (operationStyle.equals("document")) {
        return SOAPUtil.buildMessageFromDocumentSOAPBodyElement(
            inputMessage.getParts().values(), bodies[0]);
      } else {
        return SOAPUtil.buildMessageFromRpcSOAPBodyElement(
            inputMessage.getParts().values(), bodies[0]);
      }
    }
  }


  protected static class GetOperationCommand implements Command<Operation> {
    private final QName processQName;
    private final QName portTypeQName;
    private final String operationName;

    public GetOperationCommand(final QName processQName, final QName portTypeQName,
        final String operationName) {
      this.processQName = processQName;
      this.portTypeQName = portTypeQName;
      this.operationName = operationName;
    }

    public Operation execute(final Environment environment) throws Exception {
      Repository repository = environment.get(Repository.class);

      BpelProcess bpelProcess = repository.getProcess(processQName);
      WsdlsInfos wsdlsInfos = bpelProcess.getWsdlInfos();

      return getOperation(wsdlsInfos.getPortType(portTypeQName),
          operationName);

    }

  }


  public static Element[] call(final Element[] bodies, final QName processQName,
      final QName portTypeQName, final String operationName, final String operationStyle) {

    if (log.isLoggable(Level.FINE)) {
      log.fine("Entering WS Implementation, process = " + processQName
          + ", PT = " + portTypeQName + ", op = " + operationName
          + ", opStyle = " + operationStyle);
      for (Element body : bodies) {
        log.fine("body = " + body);
      }
    }
    Element[] elementsReturned = null;
    Message incomingMessage = null;

    EnvironmentFactory environmentFactory = EnvFactoryRepository.get();
    CommandService commandService = environmentFactory.get(CommandService.class);

    Operation operation =
      commandService.execute(new GetOperationCommand(processQName, portTypeQName, operationName));

    OperationKey operationKey = new OperationKey(processQName, portTypeQName,
        operationName);
    incomingMessage =
      commandService.execute(
          new BuildIncomingMessageCommand(operation, processQName, operationStyle, bodies)
      );

    if (log.isLoggable(Level.FINE)) {
      log.fine("incomingMessage = " + incomingMessage);
    }

    MessageCarrier messageCarrier = null;
    BpelExecution instance = null;

    if (operation.getOutput() != null) {
      // it is a 2 ways operation. We have to get an identifier.
      messageCarrier = new MessageCarrier();
    }
    if (log.isLoggable(Level.FINE)) {
      log.fine("Doing call to receiver, messageCarrier = " + messageCarrier);
    }
    instance = Receiver.handle(incomingMessage, operationKey,
        EnvFactoryRepository.get(), messageCarrier);
    if (log.isLoggable(Level.FINE)) {
      log.fine("call to receiver done. Instance = " + instance);
    }
    if (operation.getOutput() != null) {
      if (log.isLoggable(Level.FINE)) {
        log.fine("This operation has an output, waiting for messageCarrier...");
      }
      Message responseMessage = messageCarrier.getMessage();
      if (operationStyle.equals("document")) {
        if (log.isLoggable(Level.FINE)) {
          log.fine("This operation is using 'document' style.");
        }
        // We make the response");
        List<Part> parts = getOutputParts(operation);
        elementsReturned = new Element[parts.size()];
        int i = 0;
        for (Part part : parts) {
          elementsReturned[i] = responseMessage.getPartValue(part.getName());
          i++;
        }
      } else {
        if (log.isLoggable(Level.FINE)) {
          log.fine("This operation is using 'rpc' style.");
        }
        // it is a rpc style!!
        // We make the response");
        Document rpcDocument = SOAPUtil.buildRpcDocument(responseMessage, operationName);
        elementsReturned = new Element[1];
        elementsReturned[0] = rpcDocument.getDocumentElement();
      }
    }
    if (log.isLoggable(Level.FINE)) {
      log.fine("elementsReturned = " + Arrays.toString(elementsReturned));
    }
    if (instance != null) {
      instance = instance.getProcessInstance();
      boolean removeMutex = false;
      BpelExecution lockedInstance = instance;
      InstanceLock instanceLock = MutexRepository.lockInstance(lockedInstance);
      try {
        commandService.execute(new RemoveInstanceCommand(instance));
        removeMutex = true;
      } finally {
        if (removeMutex) {
          MutexRepository.removeAndUnlockInstance(instance.getUUID());
          if (log.isLoggable(Level.FINE)) {
            log.fine("removing instance = " + instance.getUUID());
          }
        } else {
          MutexRepository.unlockInstance(instanceLock);
        }
      }
    }
    return elementsReturned;
  }

  private static Operation getOperation(final PortType portType, final String operationName) {
    List<Operation> operations = portType.getOperations();
    for (Operation operation : operations) {
      if (operation.getName().equals(operationName)) {
        return operation;
      }
    }
    return null;
  }

  private static List<Part> getOutputParts(final Operation operation) {
    Output output = operation.getOutput();
    if (output != null) {
      return getMessageParts(output.getMessage());
    }
    return new ArrayList<Part>();
  }

  @SuppressWarnings("unchecked")
  private static List<Part> getMessageParts(final javax.wsdl.Message message) {
    Map< ? , Part> partMap = message.getParts();
    Collection<Part> parts = partMap.values();
    return new ArrayList<Part>(parts);
  }
}
