/**
 * Copyright (C) 2007  Bull S. A. S.
 * Bull, Rue Jean Jaures, B.P.68, 78340, Les Clayes-sous-Bois
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation
 * version 2.1 of the License.
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA  02110-1301, USA.
 */
package org.ow2.orchestra.axis;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.wsdl.Binding;
import javax.wsdl.BindingOperation;
import javax.wsdl.Operation;
import javax.wsdl.Part;
import javax.wsdl.Port;
import javax.wsdl.PortType;
import javax.wsdl.Service;
import javax.wsdl.extensions.ExtensibilityElement;
import javax.wsdl.extensions.soap.SOAPAddress;
import javax.wsdl.extensions.soap.SOAPBinding;
import javax.wsdl.extensions.soap.SOAPBody;
import javax.wsdl.extensions.soap.SOAPOperation;
import javax.xml.namespace.QName;
import javax.xml.soap.Name;
import javax.xml.soap.SOAPBodyElement;
import javax.xml.soap.SOAPFault;
import javax.xml.soap.SOAPMessage;

import org.jbpm.pvm.env.Environment;
import org.jbpm.pvm.internal.util.XmlUtil;
import org.ow2.orchestra.definition.BpelProcess;
import org.ow2.orchestra.facade.exception.OrchestraRuntimeException;
import org.ow2.orchestra.runtime.PartnerLinkRuntime;
import org.ow2.orchestra.runtime.VariableRuntime;
import org.ow2.orchestra.services.OperationKey;
import org.ow2.orchestra.services.itf.Invoker;
import org.ow2.orchestra.services.itf.Repository;
import org.ow2.orchestra.util.SOAPUtil;
import org.ow2.orchestra.var.Message;
import org.w3c.dom.Element;

/**
 * Invoker.java.
 * 
 * @author Goulven Le Jeune, Charles Souillard, Candy Chlad, Stanislas Giraudet
 *         De Boudemange, Guillaume Porcher
 * 
 * Created on : Jul 3, 2007
 */
public class AxisInvoker implements Invoker {

  private static Logger log = Logger.getLogger(AxisInvoker.class.getName());

  /** The QName of the soap binding element in WSDL. */
  private static final QName SOAP_BINDING = new QName(
      "http://schemas.xmlsoap.org/wsdl/soap/", "binding");

  /** The QName of the soap operation element in WSDL. */
  private static final QName SOAP_OPERATION = new QName(
      "http://schemas.xmlsoap.org/wsdl/soap/", "operation");

  /** The QName of the soap body element in WSDL. */
  private static final QName SOAP_BODY = new QName(
      "http://schemas.xmlsoap.org/wsdl/soap/", "body");

  /**
   * perform an invoke using the given parameters.
   * 
   * @param instance
   *            the instance of the process
   * @param name
   *            the name of the invoke activity
   * @param partnerLinkRuntime
   *            the partnerLink runtime value of the invoke
   * @param operation
   *            the operation of the invoke activity
   * @param variableRuntime
   *            the value of the variable used in invoke
   * @return the message received in response of this call (if there is one...)
   */
  public Message invoke(OperationKey operationKey, String name,
      PartnerLinkRuntime partnerLinkRuntime, VariableRuntime variableRuntime) {

    // TODO: exceptions
    if (log.isLoggable(Level.FINER)) {
      log.entering(AxisInvoker.class.getName(), "invoke", new Object[] {
        operationKey, name, partnerLinkRuntime, variableRuntime});
    }

    Message messageReturned = null;

    try {
      // We find the portType specified for the partner role
      QName portTypeQName = operationKey.getPortTypeQName();
      String operationName = operationKey.getOperationName();
      QName processQName = operationKey.getProcessQName();
      Repository repository = Environment.getCurrent().get(Repository.class);
      BpelProcess process = repository.getProcess(processQName);

      PortType portType = process.getWsdlInfos().getPortType(portTypeQName);

      // We find the bindings for this portType
      List<Service> services = process.getWsdlInfos().getServicesOfPortType(
          portTypeQName);
      // We take the first binding of the WSDL definition which binds the
      // portType
      // TODO: Time complexity of this algorithm is high! (O(s*p*e)) !
      final List<Binding> bindings = new ArrayList<Binding>();
      for (Service service : services) {
        for (Port port : (Collection<Port>) service.getPorts().values()) {
          List<ExtensibilityElement> elements = port.getExtensibilityElements();
          for (ExtensibilityElement element : elements) {
            if (!(element instanceof SOAPAddress)) {
              log.warning("The extensibility element: "
                  + element.getElementType() + " is not supported!");
            } else {
              if (port.getBinding().getPortType().getQName().equals(portTypeQName)) {
                bindings.add(port.getBinding());
              }
            }
          }
        }
      }
      if (bindings.size() == 0) {
        throw new OrchestraRuntimeException("Can't find any supported binding!");
      }

      final Binding binding = bindings.get(0);
      log.fine("Using the first binding supported: " + binding.getQName());

      // Get the Operation associated with the portType.
      Operation operation = portType.getOperation(operationName, null, null);
      // Get the partner endpoint adress
      String endpoint = resolveEndPoint(partnerLinkRuntime
          .getPartnerRoleEndPointReference());

      // style used for the operation (RPC or document)
      String style = null;
      // Use used for the input of the operation
      String use = null;
      String soapAction = null;

      // we look for the defaut style (Document) or (RPC) to use for operations.
      for (Object elem : binding.getExtensibilityElements()) {
        // QName elementQName = ((ExtensibilityElement) elem).getElementType();
        // if (elementQName.equals(SOAP_BINDING)) {
        if (elem instanceof SOAPBinding) {
          SOAPBinding soapBinding = (SOAPBinding) elem;
          style = soapBinding.getStyle();
          break;
        }
      }

      // get particular style configuration for operation and URI of soapAction
      BindingOperation opBinding = binding.getBindingOperation(operationName,
          null, null);
      for (Object elem : opBinding.getExtensibilityElements()) {
        // QName elementQName = ((ExtensibilityElement) elem).getElementType();
        // if (elementQName.equals(SOAP_OPERATION)) {
        if (elem instanceof SOAPOperation) {
          SOAPOperation soapOperation = (SOAPOperation) elem;
          if (soapOperation.getStyle() != null) {
            style = soapOperation.getStyle();
          }
          soapAction = soapOperation.getSoapActionURI();
          break;
        }
      }

      // get the use of the Input (Literal or encoded)
      for (Object elem : opBinding.getBindingInput().getExtensibilityElements()) {
        // QName elementQName = ((ExtensibilityElement) elem).getElementType();
        // if (elementQName.equals(SOAP_BODY)) {
        if (elem instanceof SOAPBody) {
          SOAPBody soapBody = (SOAPBody) elem;
          use = soapBody.getUse();
          break;
        }
      }

      if (soapAction == null) {
        throw new OrchestraRuntimeException(
            "No soapAction specified for this operation : " + operationName);
      }

      Map inputParts = operation.getInput().getMessage().getParts();
      if (style.equals("document") && inputParts.size() != 1) {
        throw new OrchestraRuntimeException(
            "Style is document but input message has many parts : "
            + "not supported as WSI Basic profiles requires only one part for document style");
      }
      if (log.isLoggable(Level.FINE)) {
        log.fine("Create the  call to the endPoint['" + endpoint + "']");
      }
      Message message = (Message) variableRuntime.getValue();

      SOAPMessage request = null;
      if (style.equals("document")) {
        if (message.getParts().size() != 1) {
          throw new OrchestraRuntimeException(
              "Trying to perform a call with many parts using document style : "
              + "WSI Basic profiles does not allow that : only one part is expected (having the name of the operation");
        }
        request = SOAPUtil.buildDocumentSOAPMessage(soapAction, message);
      } else {
        request = SOAPUtil.buildRpcSOAPMessage(soapAction, message,
            operationName);
      }
      SOAPMessage response = SOAPUtil.call(request, endpoint);
      if (operation.getOutput() != null) {
        Map outputParts = operation.getOutput().getMessage().getParts();
        if (style.equals("document") && outputParts.size() != 1) {
          throw new OrchestraRuntimeException(
              "Style is document but output message has many parts : "
              + "not supported as WSI Basic profiles requires only one part for document style");
        }
        if (response == null) {
          throw new OrchestraRuntimeException("This operation(" + binding.getQName()
              + ":" + operationName
              + ") requires a response but no one was received");
        }
        if (response.getSOAPBody().hasFault()) {
          SOAPFault newFault = response.getSOAPBody().getFault();
          Name code = newFault.getFaultCodeAsName();
          String string = newFault.getFaultString();
          String actor = newFault.getFaultActor();
          throw new OrchestraRuntimeException("SOAP call returned a fault: " + string);
        }
        
        messageReturned = getMessageSOAPFromResponse(response, outputParts,
            style, use);
      }
    } catch (OrchestraRuntimeException oe) {
      throw oe;
    } catch (Exception e) {
      throw new OrchestraRuntimeException("Exception caught while invoke WS", e);
    }
    return messageReturned;
  }

  private Message getMessageSOAPFromResponse(SOAPMessage response,
      Map<String, Part> outputParts, String style, String use) {

    try {
      Iterator<javax.xml.soap.SOAPBodyElement> bodyElementsIterator = response
      .getSOAPBody().getChildElements();
      if (!bodyElementsIterator.hasNext()) {
        throw new OrchestraRuntimeException(
            "Exception caught while building a message from soapResponse "
            + "(one and only one element was expected as a child of body but 0 were received) : "
            + response);
      }
      SOAPBodyElement element = bodyElementsIterator.next();
      if (bodyElementsIterator.hasNext()) {
        throw new OrchestraRuntimeException(
            "Exception caught while building a message from soapResponse "
            + "(only one element was expected as a child of body but many were received) : "
            + response);
      }
      if (style.equals("document")) {
        return SOAPUtil.buildMessageFromDocumentSOAPBodyElement(outputParts
            .values(), element);
      } else {
        return SOAPUtil.buildMessageFromRpcSOAPBodyElement(
            outputParts.values(), element);
      }

    } catch (Exception e) {
      throw new OrchestraRuntimeException(
          "Exception caught while building a message from soapResponse : "
          + response, e);
    }
  }

  /**
   * Extract the endPoint reference from a service reference information item.
   * 
   * @return the end Point URL
   */
  public static String resolveEndPoint(final Element serviceRefElem) {
    org.w3c.dom.Element endpointRef = XmlUtil.element(serviceRefElem,"EndpointReference");
    org.w3c.dom.Element address = XmlUtil.element(endpointRef,"Address");
    return address.getTextContent();
  }

}
