package org.ow2.orchestra.axis;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.List;

import javax.wsdl.Binding;
import javax.wsdl.BindingOperation;
import javax.wsdl.Input;
import javax.wsdl.Operation;
import javax.wsdl.Output;
import javax.wsdl.PortType;
import javax.wsdl.extensions.ExtensibilityElement;
import javax.xml.namespace.QName;

import org.ow2.orchestra.facade.exception.OrchestraRuntimeException;
import org.ow2.orchestra.ws.WSDeployer;

/**
 * This class generates a java class that implements a wsdl binding.
 *
 * BindingFileWriter.java
 *
 * @author Charles Souillard
 *
 * Created on : Mar 12, 2007
 */
public class BindingFileWriter {

  /** the binding which is implemented. */
  private Binding binding;

  /** path to the directory where wsdl2java generated files could be found. */
  private File baseDir;

  /** processName of the process which is described by the current wsdl. */
  private QName processQName;

  /** a bufferWriter which will be used in most of the methods of the class. */
  private BufferedWriter bw;

  /** PortType implemented. */
  private PortType portType;

  /** Name of this class. */
  private String className;

  public BindingFileWriter(final File baseDir, final Binding binding,
      final QName processQName) {
    this.baseDir = baseDir;
    this.binding = binding;
    this.processQName = processQName;
  }

  /**
   * Writes the binding implementation.
   *
   * @return the generated class name.
   */
  public String write() {
    QName bindingQN = this.binding.getQName();
    this.className = bindingQN.getLocalPart() + "Impl";
    String firstLetter = this.className.substring(0, 1);
    String nextLetters = this.className.substring(1);
    this.className = firstLetter.toUpperCase() + nextLetters;

    String bindingPackage = AxisDeployer.getPackageFromNamespace(bindingQN
        .getNamespaceURI());
    String bindingDir = AxisDeployer.getDirectoryFromPackage(bindingPackage);
    String bindingFilePath = this.baseDir + File.separator + bindingDir
      + File.separator + className + ".java";
    File fout = new File(bindingFilePath);
    replaceFile(fout);

    portType = this.binding.getPortType();

    try {
      FileWriter fw = new FileWriter(fout);
      this.bw = new BufferedWriter(fw);

      write("package " + bindingPackage + ";\n");

      // xml
      addImport(org.w3c.dom.Element.class);
      addImport(QName.class);
      addImport(AxisWSImpl.class);

      write("\n");
      write("public class " + this.className + " {\n");

      List<Operation> operations = (List<Operation>) portType.getOperations();
      write("  QName processQName = new QName(\""
          + this.processQName.getNamespaceURI() + "\", \""
          + this.processQName.getLocalPart() + "\");");
      write("  QName portTypeQName = new QName(\""
          + this.portType.getQName().getNamespaceURI() + "\", \""
          + this.portType.getQName().getLocalPart() + "\");");
      write("\n");
      addOperations(operations);
      write("}\n");


    } catch (Exception e) {
      throw new OrchestraRuntimeException(e);
    } finally {
      if (this.bw != null) {
        try {
          this.bw.close();
        } catch (Exception e) {
          // nothing
        }
      }
    }
    return bindingPackage + "." + this.className;
  }

  private void addOperations(final List<Operation> operations) {
    for (Operation operation : operations) {
      write("  public Element [] " + operation.getName() + "(Element[] bodies) {");
      write("    String operationName = \"" + operation.getName() + "\";");
      String operationStyle = getOperationStyle(operation);
      write("    String operationStyle = \"" + operationStyle + "\";");

      write("    return AxisWSImpl.call(bodies, processQName, portTypeQName, operationName, operationStyle);");

      write("  }");
    }
  }

  private String getOperationStyle(final Operation operation) {
    Input input = operation.getInput();
    String inputName = null;
    if (input != null) {
      inputName = input.getName();
    }
    String outputName = null;
    Output output = operation.getOutput();
    if (output != null) {
      outputName = output.getName();
    }
    BindingOperation bindingOperation = this.binding.getBindingOperation(
        operation.getName(), inputName, outputName);

    ExtensibilityElement soapBinding = WSDeployer
    .getSoapBindingFromBinding(binding);
    String soapBindingStyle = WSDeployer.getSOAPStyle(soapBinding);
    if (soapBindingStyle == null) {
      soapBindingStyle = WSDeployer.DEFAULT_SOAP_BINDING_STYLE;
    }
    ExtensibilityElement soapOperation = WSDeployer
    .getSoapOperationFromBindingOperation(bindingOperation);
    String operationStyle = soapBindingStyle;
    if (soapOperation != null) {
      String operationStyle2 = WSDeployer.getSOAPStyle(soapOperation);
      if (operationStyle2 != null) {
        operationStyle = operationStyle2;
      }
    }
    return operationStyle;
  }

  /**
   * This method delete the given file if it exists and then create a new File
   * instead.
   *
   * @param f -
   *            file to replace.
   */
  private void replaceFile(final File f) {
    // Create path to file
    f.getParentFile().mkdirs();
    if (f.exists() && !f.delete()) {
      throw new OrchestraRuntimeException(this.getClass() + ".replaceFile, cannot delete file : " + f);
    }
    boolean created;
    try {
      created = f.createNewFile();
    } catch (IOException e) {
      throw new OrchestraRuntimeException(e);
    }
    if (!created) {
      throw new OrchestraRuntimeException(this.getClass() + ".replaceFile, cannot create file : " + f);
    }
  }

  private void addImport(Class< ? > clazz) {
    write("import " + clazz.getName() + ";");
  }

  /**
   * Utility method which writes the given string in the StringBuffer.
   *
   * @param s -
   *            the string to write
   */
  private void write(final String s) {
    try {
      this.bw.write(s + "\n");
    } catch (IOException e) {
      throw new OrchestraRuntimeException(e);
    }
  }

  /**
   * Utility method which writes a new line in the StringBuffer.
   */
  private void writeNewLine() {
    try {
      this.bw.write("\n");
    } catch (IOException e) {
      throw new OrchestraRuntimeException(e);
    }
  }
}
