/**
 * Copyright (C) 2006  Bull S. A. S.
 * Bull, Rue Jean Jaures, B.P.68, 78340, Les Clayes-sous-Bois
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation
 * version 2.1 of the License.
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA  02110-1301, USA.
 **/
package org.ow2.orchestra.axis;

import javax.xml.namespace.QName;

import org.apache.axis.AxisEngine;
import org.apache.axis.AxisFault;
import org.apache.axis.MessageContext;
import org.apache.axis.description.OperationDesc;
import org.apache.axis.handlers.soap.SOAPService;
import org.apache.axis.message.SOAPEnvelope;
import org.apache.axis.providers.java.JavaProvider;
import org.apache.axis.providers.java.MsgProvider;
import org.apache.axis.utils.ClassUtils;
import org.apache.axis.utils.Messages;
import org.apache.axis.utils.cache.ClassCache;
import org.apache.axis.utils.cache.JavaClass;

/**
 * BPELMsgProvider.java
 *
 * @author Goulven Le Jeune & Charles Souillard
 * @version $Id:ProcessMsgProvider.java 813 2007-10-10 08:21:36Z souillac $
 *
 */
public class ProcessMsgProvider extends MsgProvider {

  private static final long serialVersionUID = 5240751552526750450L;

  private static final String METHOD_BODYARRAY = "soapbodyelement";

  private static final String METHOD_ELEMENTARRAY = "element";

  private static final String METHOD_DOCUMENT = "document";

  /*
   * (non-Javadoc)
   *
   * @see org.apache.axis.providers.java.JavaProvider#makeNewServiceObject(org.apache.axis.MessageContext,
   *      java.lang.String)
   */
  @Override
  protected Object makeNewServiceObject(final MessageContext msgContext,
      final String clsName) throws Exception {
    final String processName = (String) msgContext.getService().getOption(
        "bpelProcessName");
    final String processNamespace = (String) msgContext.getService().getOption(
        "bpelProcessNamespace");
    final QName processQName = new QName(processNamespace, processName);
    final ClassLoader cl = ClassLoaderMap.getCL(processQName);
    final ClassCache cache = msgContext.getAxisEngine().getClassCache();
    final JavaClass jc = cache.lookup(clsName, cl);
    return jc.getJavaClass().newInstance();
  }

  /**
   * Returns the Class info about the service class.
   */
  @Override
  protected Class getServiceClass(final String clsName, final SOAPService service,
      final MessageContext msgContext) throws AxisFault {
    ClassLoader cl = null;
    Class serviceClass = null;

    final String processName = (String) service.getOption("bpelProcessName");
    final String processNamespace = (String) service
        .getOption("bpelProcessNamespace");
    final QName processQName = new QName(processNamespace, processName);
    cl = ClassLoaderMap.getCL(processQName);

    final AxisEngine engine = service.getEngine();
    // If we have an engine, use its class cache
    if (engine != null) {
      final ClassCache cache = engine.getClassCache();
      try {
        final JavaClass jc = cache.lookup(clsName, cl);
        serviceClass = jc.getJavaClass();
      } catch (final ClassNotFoundException e) {
        JavaProvider.log.error(Messages.getMessage("exception00"), e);
        throw new AxisFault(
            Messages.getMessage("noClassForService00", clsName), e);
      }
    } else {
      // if no engine, we don't have a cache, use Class.forName instead.
      try {
        serviceClass = ClassUtils.forName(clsName, true, cl);
      } catch (final ClassNotFoundException e) {
        JavaProvider.log.error(Messages.getMessage("exception00"), e);
        throw new AxisFault(
            Messages.getMessage("noClassForService00", clsName), e);
      }
    }
    return serviceClass;
  }


  @Override
  public void processMessage(final MessageContext msgContext, final SOAPEnvelope reqEnv,
      final SOAPEnvelope resEnv, final Object obj) throws Exception {
    final ClassLoader old = Thread.currentThread().getContextClassLoader();
    final ClassLoader cl = obj.getClass().getClassLoader();
    try {
      Thread.currentThread().setContextClassLoader(cl);
      super.processMessage(msgContext, reqEnv, resEnv, obj);
    } finally {
      Thread.currentThread().setContextClassLoader(old);
    }
  }

  protected void setOperationStyle(final MessageContext msgContext) {

    final OperationDesc operation = msgContext.getOperation();
    if (operation == null || operation.getMessageOperationStyle() > -1) {
      return;
    }
    int methodType = operation.getMessageOperationStyle();
    final String methodSignature = operation.getMethod().toString().toLowerCase();
    if (methodSignature.indexOf(ProcessMsgProvider.METHOD_BODYARRAY) != -1) {
      methodType = OperationDesc.MSG_METHOD_BODYARRAY;
    } else if (methodSignature.indexOf(ProcessMsgProvider.METHOD_ELEMENTARRAY) != -1) {
      methodType = OperationDesc.MSG_METHOD_ELEMENTARRAY;
    } else if (methodSignature.indexOf(ProcessMsgProvider.METHOD_DOCUMENT) != -1) {
      methodType = OperationDesc.MSG_METHOD_DOCUMENT;
    } else {
      methodType = OperationDesc.MSG_METHOD_SOAPENVELOPE;
    }
    operation.setMessageOperationStyle(methodType);
  }

  @Override
  public void invoke(final MessageContext msgContext) throws AxisFault {
    // Make sure that the method style is correctly set (This does not appear to
    // be handled by default)
    this.setOperationStyle(msgContext);
    super.invoke(msgContext);
  }

}
