package org.ow2.orchestra.axis;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Properties;

import javax.servlet.Servlet;
import javax.wsdl.Service;
import javax.xml.namespace.QName;

import org.ow2.orchestra.definition.BpelProcess;
import org.ow2.orchestra.facade.exception.OrchestraRuntimeException;
import org.ow2.orchestra.pvm.env.Environment;
import org.ow2.orchestra.services.OperationKey;
import org.ow2.orchestra.services.impl.PublisherImpl;
import org.ow2.orchestra.services.itf.HttpPublisher;
import org.ow2.orchestra.wsdl.WsdlsInfos;

/**
 * AxisServicePublisher.java. Implementation of the service publisher with axis.
 *
 * @author Charles Souillard
 *
 * Created on : Jun 26, 2007
 */
public class AxisPublisher extends PublisherImpl implements HttpPublisher {

  @Override
  public void publishServices(final BpelProcess bpelProcess, final Environment environment) {
    super.publishServices(bpelProcess, environment);
    final AxisDeployer axisDeployer = new AxisDeployer(bpelProcess);
    final List<Service> services = this.getServices(bpelProcess);
    if (services != null) {
      axisDeployer.deploy(services);
    }
  }

  @Override
  public void unpublishServices(final BpelProcess bpelProcess, final Environment environment) {
    final AxisDeployer axisDeployer = new AxisDeployer(bpelProcess);
    final List<Service> services = this.getServices(bpelProcess);
    if (services != null) {
      axisDeployer.undeploy(services);
    }
    super.unpublishServices(bpelProcess, environment);
  }

  private List<Service> getServices(final BpelProcess bpelProcess) {
    // get operation keys for receiving activities
    final Collection<OperationKey> opKeys = this.getServiceOperationKeys(bpelProcess, bpelProcess.getInitial(), Environment.getCurrent());

    final WsdlsInfos wsdlInfos = bpelProcess.getWsdlInfos();
    List<Service> services = null;

    // to store portTypes of receving activities that does not have a service associated
    final List<QName> missingServices = new ArrayList<QName>();

    // try to find a WSDL service to deploy for each receiving activity
    for (final OperationKey operationKey : opKeys) {
      final QName portTypeQName = operationKey.getPortTypeQName();
      final List<Service> portTypeServices = wsdlInfos.getServicesOfPortType(portTypeQName);
      if (portTypeServices != null) {
        if (services == null) {
          services = new ArrayList<Service>();
        }
        for (final Service service : portTypeServices) {
          if (!services.contains(service)) {
            services.add(service);
          }
        }
      } else {
        // no service found for port type
        missingServices.add(portTypeQName);
      }
    }

    // throw exception if there are missing services.
    if (!missingServices.isEmpty()) {
      final StringBuffer msg = new StringBuffer("Some wsdl services not found for receiving activities. "
          + "Please specify service for the following PortTypes: ");
      boolean first = true;
      for (final QName ptQName : missingServices) {
        msg.append(first ? "" : ", ").append(ptQName.toString());
        first = false;
      }
      msg.append(".");
      throw new OrchestraRuntimeException(msg.toString());
    }
    return services;
  }
  /* (non-Javadoc)
   * @see org.ow2.orchestra.services.itf.HttpPublisher#getPublisherServlet()
   */
  public Servlet getPublisherServlet(final Properties orchestraProperties) {
    return new OrchestraBaseAxisServlet();
  }
}
