/**
 * Copyright (C) 2006  Bull S. A. S.
 * Bull, Rue Jean Jaures, B.P.68, 78340, Les Clayes-sous-Bois
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation
 * version 2.1 of the License.
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA  02110-1301, USA.
 **/
package org.ow2.orchestra.axis;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.wsdl.Fault;
import javax.wsdl.Operation;
import javax.wsdl.Output;
import javax.wsdl.Part;
import javax.wsdl.PortType;
import javax.xml.namespace.QName;

import org.apache.axis.AxisFault;
import org.apache.axis.management.ServiceAdmin;
import org.apache.axis.server.AxisServer;
import org.ow2.orchestra.definition.BpelProcess;
import org.ow2.orchestra.facade.exception.OrchestraRuntimeException;
import org.ow2.orchestra.pvm.env.Environment;
import org.ow2.orchestra.pvm.internal.cmd.Command;
import org.ow2.orchestra.pvm.internal.cmd.CommandService;
import org.ow2.orchestra.runtime.BpelExecution;
import org.ow2.orchestra.services.MessageCarrierImpl;
import org.ow2.orchestra.services.OperationKey;
import org.ow2.orchestra.services.ReceivingService;
import org.ow2.orchestra.services.itf.Repository;
import org.ow2.orchestra.util.BpelSOAPUtil;
import org.ow2.orchestra.var.MessageVariable;
import org.ow2.orchestra.wsdl.WsdlsInfos;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

public final class AxisWSImpl {

  private static Logger log = Logger.getLogger(AxisWSImpl.class.getName());

  private final QName processQName;
  private final QName portTypeQName;

  private final Map<String,Operation> operations = new HashMap<String, Operation>();

  private final AxisServer engine;

  public AxisWSImpl(final QName processQName, final QName portTypeQName) {
    this.processQName = processQName;
    this.portTypeQName = portTypeQName;
    try {
      this.engine = ServiceAdmin.getEngine();
    } catch (final AxisFault e) {
      throw new OrchestraRuntimeException("Exception during Web service deployment.", e);
    }
  }

  protected static class GetOperationCommand implements Command<Operation> {

    private static final long serialVersionUID = 5451904601397581112L;

    private final QName processQName;
    private final QName portTypeQName;
    private final String operationName;

    public GetOperationCommand(final QName processQName, final QName portTypeQName, final String operationName) {
      this.processQName = processQName;
      this.portTypeQName = portTypeQName;
      this.operationName = operationName;
    }

    public Operation execute(final Environment environment) {
      final Repository repository = environment.get(Repository.class);

      final List<BpelProcess> bpelProcesses = repository.getProcesses(this.processQName);
      for (final BpelProcess bpelProcess : bpelProcesses) {
        final WsdlsInfos wsdlsInfos = bpelProcess.getWsdlInfos();
        final PortType portType = wsdlsInfos.getPortType(this.portTypeQName);
        if (portType != null) {
          final Operation op = AxisWSImpl.getOperation(portType, this.operationName);
          if (op != null) {
            return op;
          }
        }
      }
      return null;
    }

  }

  public Operation getOperation(final String operationName) {
    if (!this.operations.containsKey(operationName)) {
      final CommandService commandService = (CommandService) this.engine.getOption(CommandService.class.getName());
      final Operation op = commandService.execute(
          new GetOperationCommand(
            this.processQName,
            this.portTypeQName,
            operationName));
      this.operations.put(operationName, op);
    }
    return this.operations.get(operationName);
  }


  public Element[] call(final Element[] bodies,
      final String operationName,
      final String operationStyle, final boolean lock) throws AxisFault {

    if (AxisWSImpl.log.isLoggable(Level.FINE)) {
      AxisWSImpl.log.fine("Entering WS Implementation, process = "
          + this.processQName
          + ", PT = " + this.portTypeQName
          + ", op = " + operationName
          + ", opStyle = " + operationStyle);
      for (final Element body : bodies) {
        AxisWSImpl.log.fine("body = " + body);
      }
    }
    Element[] elementsReturned = null;
    MessageVariable incomingMessage = null;

    final Operation operation = this.getOperation(operationName);
    final javax.wsdl.Message inputMessage = operation.getInput().getMessage();
    if (operationStyle.equals("document")) {
      incomingMessage = BpelSOAPUtil.buildMessageFromDocumentSOAPBodyElement(inputMessage.getParts().values(), bodies[0]);
    } else {
      incomingMessage = BpelSOAPUtil.buildMessageFromRpcSOAPBodyElement(inputMessage.getParts().values(), bodies[0]);
    }

    if (AxisWSImpl.log.isLoggable(Level.FINE)) {
      AxisWSImpl.log.fine("incomingMessage = " + incomingMessage);
    }

    MessageCarrierImpl messageCarrier = null;
    BpelExecution instance = null;

    if (operation.getOutput() != null) {
      // it is a 2 ways operation. We have to get an identifier.
      messageCarrier = new MessageCarrierImpl();
    }
    if (AxisWSImpl.log.isLoggable(Level.FINE)) {
      AxisWSImpl.log.fine("Doing call to receiver, messageCarrier = " + messageCarrier);
    }
    final OperationKey operationKey = new OperationKey(this.processQName, this.portTypeQName, operationName);
    final CommandService commandService = (CommandService) this.engine.getOption(CommandService.class.getName());
    instance = ReceivingService.handle(incomingMessage, operationKey, messageCarrier, lock, commandService);
    if (AxisWSImpl.log.isLoggable(Level.FINE)) {
      AxisWSImpl.log.fine("call to receiver done. Instance = " + instance);
    }
    if (operation.getOutput() != null) {
      if (AxisWSImpl.log.isLoggable(Level.FINE)) {
        AxisWSImpl.log.fine("This operation has an output, waiting for messageCarrier...");
      }
      final MessageVariable responseMessage = messageCarrier.getMessage();
      final QName faultQName = messageCarrier.getFaultQName();
      if (faultQName != null && faultQName.getNamespaceURI().equals(this.portTypeQName.getNamespaceURI())) {
        // We make the fault detail;
        final List<Part> parts = AxisWSImpl.getFaultParts(operation, faultQName.getLocalPart());
        elementsReturned = new Element[parts.size()];
        int i = 0;
        for (final Part part : parts) {
          elementsReturned[i] = responseMessage.getPartValue(part.getName());
          i++;
        }
        // throw fault
        final AxisFault fault = new AxisFault();
        fault.setFaultCode(faultQName);
        fault.setFaultDetail(elementsReturned);
        throw fault;
      }
      if (operationStyle.equals("document")) {
        if (AxisWSImpl.log.isLoggable(Level.FINE)) {
          AxisWSImpl.log.fine("This operation is using 'document' style.");
        }
        // We make the response");
        final List<Part> parts = AxisWSImpl.getOutputParts(operation);
        elementsReturned = new Element[parts.size()];
        int i = 0;
        for (final Part part : parts) {
          elementsReturned[i] = responseMessage.getPartValue(part.getName());
          i++;
        }
      } else {
        if (AxisWSImpl.log.isLoggable(Level.FINE)) {
          AxisWSImpl.log.fine("This operation is using 'rpc' style.");
        }
        // it is a rpc style!!
        // We make the response");
        // Add response to operation name to conform with WS-Basic profile R2729
        final Document rpcDocument = BpelSOAPUtil.buildRpcDocument(responseMessage, operationKey.getOperationName() + "Response");
        elementsReturned = new Element[1];
        elementsReturned[0] = rpcDocument.getDocumentElement();
      }

    }
    if (AxisWSImpl.log.isLoggable(Level.FINE)) {
      AxisWSImpl.log.fine("elementsReturned = " + Arrays.toString(elementsReturned));
    }
    return elementsReturned;
  }

  private static Operation getOperation(final PortType portType, final String operationName) {
    final List<Operation> operations = portType.getOperations();
    for (final Operation operation : operations) {
      if (operation.getName().equals(operationName)) {
        return operation;
      }
    }
    return null;
  }

  private static List<Part> getOutputParts(final Operation operation) {
    final Output output = operation.getOutput();
    if (output != null) {
      return AxisWSImpl.getMessageParts(output.getMessage());
    }
    return new ArrayList<Part>();
  }

  private static List<Part> getFaultParts(final Operation operation, final String faultName) {
    final Fault output = operation.getFault(faultName);
    if (output != null) {
      return AxisWSImpl.getMessageParts(output.getMessage());
    }
    return new ArrayList<Part>();
  }

  @SuppressWarnings("unchecked")
  private static List<Part> getMessageParts(final javax.wsdl.Message message) {
    final Map< ? , Part> partMap = message.getParts();
    final Collection<Part> parts = partMap.values();
    return new ArrayList<Part>(parts);
  }
}
