package org.ow2.orchestra.axis;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.List;

import javax.wsdl.Binding;
import javax.wsdl.BindingOperation;
import javax.wsdl.Input;
import javax.wsdl.Operation;
import javax.wsdl.Output;
import javax.wsdl.PortType;
import javax.wsdl.extensions.ExtensibilityElement;
import javax.xml.namespace.QName;

import org.apache.axis.AxisFault;
import org.ow2.orchestra.definition.BpelProcess;
import org.ow2.orchestra.facade.exception.OrchestraRuntimeException;
import org.ow2.orchestra.facade.uuid.ProcessDefinitionUUID;
import org.ow2.orchestra.services.OperationKey;
import org.ow2.orchestra.ws.WSDeployer;

/**
 * This class generates a java class that implements a wsdl binding.
 *
 * BindingFileWriter.java
 *
 * @author Charles Souillard
 *
 * Created on : Mar 12, 2007
 */
public class BindingFileWriter {

  /** the binding which is implemented. */
  private final Binding binding;

  /** path to the directory where wsdl2java generated files could be found. */
  private final File baseDir;

  /** a bufferWriter which will be used in most of the methods of the class. */
  private BufferedWriter bw;

  /** PortType implemented. */
  private PortType portType;

  /** Name of this class. */
  private String className;

  private final BpelProcess bpelProcess;

  public BindingFileWriter(final File baseDir, final Binding binding, final BpelProcess bpelProcess) {
    this.baseDir = baseDir;
    this.binding = binding;
    this.bpelProcess = bpelProcess;
  }

  /**
   * Writes the binding implementation.
   *
   * @return the generated class name.
   */
  public String write() {
    final QName bindingQN = this.binding.getQName();
    this.className = bindingQN.getLocalPart() + "Impl";
    final String firstLetter = this.className.substring(0, 1);
    final String nextLetters = this.className.substring(1);
    this.className = firstLetter.toUpperCase() + nextLetters;

    final String bindingPackage = AxisDeployer.getPackageFromNamespace(bindingQN
        .getNamespaceURI());
    final String bindingDir = AxisDeployer.getDirectoryFromPackage(bindingPackage);
    final String bindingFilePath = this.baseDir + File.separator + bindingDir
      + File.separator + this.className + ".java";
    final File fout = new File(bindingFilePath);
    this.replaceFile(fout);

    this.portType = this.binding.getPortType();

    try {
      final FileWriter fw = new FileWriter(fout);
      this.bw = new BufferedWriter(fw);

      this.write("package " + bindingPackage + ";\n");

      // xml
      this.addImport(org.w3c.dom.Element.class);
      this.addImport(QName.class);
      this.addImport(ProcessDefinitionUUID.class);
      this.addImport(AxisWSImpl.class);


      this.write("\n");
      this.write("public class " + this.className + " {\n");

      final List<Operation> operations = this.portType.getOperations();
      this.write("  QName processQName = new QName(\""
          + this.bpelProcess.getTargetNamespace() + "\", \""
          + this.bpelProcess.getName() + "\");");
      this.write("  QName portTypeQName = new QName(\""
          + this.portType.getQName().getNamespaceURI() + "\", \""
          + this.portType.getQName().getLocalPart() + "\");");
      this.write("  AxisWSImpl axisWSImpl = new AxisWSImpl(processQName, portTypeQName);");
      this.write("\n");

      this.addOperations(operations);
      this.write("}\n");


    } catch (final Exception e) {
      throw new OrchestraRuntimeException(e);
    } finally {
      if (this.bw != null) {
        try {
          this.bw.close();
        } catch (final Exception e) {
          // nothing
        }
      }
    }
    return File.separator + bindingDir + File.separator + this.className + ".java";
  }

  private void addOperations(final List<Operation> operations) {

    for (final Operation operation : operations) {
      final boolean lock =
        this.bpelProcess.getNumberOfStartElements() >  1
        && this.bpelProcess.getStartElement(
          new OperationKey(this.bpelProcess.getQName(), this.portType.getQName(), operation.getName())) != null;
      this.write("  public Element [] " + operation.getName() + "(Element[] bodies) throws " + AxisFault.class.getName() + " {");
      this.write("    String operationName = \"" + operation.getName() + "\";");
      final String operationStyle = this.getOperationStyle(operation);
      this.write("    String operationStyle = \"" + operationStyle + "\";");
      this.write("    return axisWSImpl.call(bodies, operationName, operationStyle, " + lock + ");");

      this.write("  }");
    }
  }

  private String getOperationStyle(final Operation operation) {
    final Input input = operation.getInput();
    String inputName = null;
    if (input != null) {
      inputName = input.getName();
    }
    String outputName = null;
    final Output output = operation.getOutput();
    if (output != null) {
      outputName = output.getName();
    }
    final BindingOperation bindingOperation = this.binding.getBindingOperation(
        operation.getName(), inputName, outputName);

    final ExtensibilityElement soapBinding = WSDeployer
    .getSoapBindingFromBinding(this.binding);
    String soapBindingStyle = WSDeployer.getSOAPStyle(soapBinding);
    if (soapBindingStyle == null) {
      soapBindingStyle = WSDeployer.DEFAULT_SOAP_BINDING_STYLE;
    }
    final ExtensibilityElement soapOperation = WSDeployer
    .getSoapOperationFromBindingOperation(bindingOperation);
    String operationStyle = soapBindingStyle;
    if (soapOperation != null) {
      final String operationStyle2 = WSDeployer.getSOAPStyle(soapOperation);
      if (operationStyle2 != null) {
        operationStyle = operationStyle2;
      }
    }
    return operationStyle;
  }

  /**
   * This method delete the given file if it exists and then create a new File
   * instead.
   *
   * @param f -
   *            file to replace.
   */
  private void replaceFile(final File f) {
    // Create path to file
    f.getParentFile().mkdirs();
    if (f.exists() && !f.delete()) {
      throw new OrchestraRuntimeException(this.getClass() + ".replaceFile, cannot delete file : " + f);
    }
    boolean created;
    try {
      created = f.createNewFile();
    } catch (final IOException e) {
      throw new OrchestraRuntimeException(e);
    }
    if (!created) {
      throw new OrchestraRuntimeException(this.getClass() + ".replaceFile, cannot create file : " + f);
    }
  }

  private void addImport(final Class< ? > clazz) {
    this.write("import " + clazz.getName() + ";");
  }

  /**
   * Utility method which writes the given string in the StringBuffer.
   *
   * @param s -
   *            the string to write
   */
  private void write(final String s) {
    try {
      this.bw.write(s + "\n");
    } catch (final IOException e) {
      throw new OrchestraRuntimeException(e);
    }
  }

}
