package org.ow2.orchestra.axis;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.StringWriter;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.wsdl.Binding;
import javax.wsdl.Definition;
import javax.wsdl.Import;
import javax.wsdl.Operation;
import javax.wsdl.OperationType;
import javax.wsdl.Port;
import javax.wsdl.Service;
import javax.wsdl.extensions.ExtensibilityElement;
import javax.wsdl.extensions.soap.SOAPBinding;
import javax.wsdl.extensions.soap.SOAPBody;
import javax.wsdl.extensions.soap.SOAPFault;
import javax.wsdl.extensions.soap12.SOAP12Binding;
import javax.wsdl.extensions.soap12.SOAP12Body;
import javax.wsdl.extensions.soap12.SOAP12Fault;
import javax.xml.namespace.QName;

import org.apache.axis.EngineConfiguration;
import org.apache.axis.WSDDEngineConfiguration;
import org.apache.axis.deployment.wsdd.WSDDConstants;
import org.apache.axis.deployment.wsdd.WSDDDeployment;
import org.apache.axis.deployment.wsdd.WSDDDocument;
import org.apache.axis.deployment.wsdd.WSDDException;
import org.apache.axis.deployment.wsdd.WSDDProvider;
import org.apache.axis.deployment.wsdd.WSDDService;
import org.apache.axis.management.ServiceAdmin;
import org.apache.axis.server.AxisServer;
import org.apache.axis.utils.ClassUtils;
import org.apache.axis.wsdl.toJava.Utils;
import org.ow2.jonas.eclipse.compiler.CompilationContext;
import org.ow2.jonas.eclipse.compiler.CompilerError;
import org.ow2.jonas.eclipse.compiler.JOnASCompiler;
import org.ow2.orchestra.definition.BpelProcess;
import org.ow2.orchestra.facade.Deployment;
import org.ow2.orchestra.facade.exception.OrchestraRuntimeException;
import org.ow2.orchestra.pvm.env.Environment;
import org.ow2.orchestra.pvm.internal.cmd.CommandService;
import org.ow2.orchestra.util.Misc;
import org.ow2.orchestra.util.XmlUtil;
import org.ow2.orchestra.util.wsdl.WsdlUtil;
import org.ow2.orchestra.ws.WSDeployer;

/**
 * Axis engine WS deployer.
 */
public class AxisDeployer extends WSDeployer {

  // WSDD constants
  /** deploy.wsdd. */
  public static final String WSDD_DEPLOY_FILE_NAME = "deploy.wsdd";
  public static final String WSDL_FILE_NAME = ".wsdl";

  /** undeploy.wsdd. */
  public static final String WSDD_UNDEPLOY_FILE_NAME = "undeploy.wsdd";

  /** axis deployment namespace. */
  public static final String WSDD_NS = "http://xml.apache.org/axis/wsdd/";

  /** wsdd root element name. */
  public static final String WSDD_SERVICE = "service";

  /** wsdd root element style attribute name. */
  public static final String WSDD_SERVICE_STYLE_ATTR = "style";

  /** wsdd root element provider attribute name. */
  public static final String WSDD_SERVICE_PROVIDER_ATTR = "provider";

  /** wsdd style attribute message value. */
  public static final String MESSAGE_STYLE = "Message";

  /** utility var to rename wsdd file with a suffix. */
  public static final String NEW_WSDD_SUFFIX = ".message";

  /** scope of the wsdd generated. */
  public static final String SCOPE = "Request";

  /** jdkVersion used to compile generated java files. */
  public static final String JDK_VERSION = "1.5";

  /** utility variable to define ant environment. */
  public static final String ANT_ENV = "antEnv";

  /** log. */
  private static Logger log = Logger.getLogger(Deployment.class.getName());

  private static File webAppTmpDir;

  private final File servicesDir;

  private final AxisPublisher axisPublisher;

  /**
   * Default constructor.
   *
   * @param wsId - id of the WS : in our case, processName
   * @param wsdlUrl - url of the wsdl file to deploy.
   * @param orchestraDirectory - Absolute path to the orchestra directory.
   */
  public AxisDeployer(final BpelProcess bpelProcess, final AxisPublisher axisPublisher) {
    super(bpelProcess);
    this.servicesDir = this.getServicesDir();
    this.axisPublisher = axisPublisher;
  }

  /**
   * return null if operation style is supported by Axis engine. Else, returns a
   * message explaining the problem.
   *
   * @param operationStyle - operationStyle to check
   * @return null if operation style is supported by Axis engine. Else, returns
   *         a message explaining the problem.
   */
  @Override
  protected String checkOperationStyle(final String operationStyle) {
    if (operationStyle == null) {
      return "Style attribute of this operation must be specified";
    } else if (!operationStyle.equals("document") && !operationStyle.equals("rpc")) {
      return "Style attribute of this operation must be : document or rpc";
    }
    return null;
  }

  /**
   * return null if operation type is supported by Axis engine. Else, returns a
   * message explaining the problem.
   *
   * @param operationType - operationType to check
   * @return null if operation type is supported by Axis engine. Else, returns a
   *         message explaining the problem.
   */
  @Override
  protected String checkOperationType(final OperationType operationType) {
    if (!operationType.equals(OperationType.REQUEST_RESPONSE) && !operationType.equals(OperationType.ONE_WAY)) {
      return "Operation type : " + operationType + " is not supported."
        + " Please use one of : " + OperationType.ONE_WAY + "/" + OperationType.REQUEST_RESPONSE;
    }
    return null;
  }

  /**
   * return null if soapVersion is supported by Axis engine. Else, returns a
   * message explaining the problem.
   *
   * @param soapBinding - soapBinding to check
   * @return null if soapVersion is supported by Axis engine. Else, returns a
   *         message explaining the problem.
   */
  @Override
  protected String checkSoapVersion(final ExtensibilityElement soapBinding) {
    if (!(soapBinding instanceof SOAPBinding) && !(soapBinding instanceof SOAP12Binding)) {
      return "Supported Soap Version are " + WSDeployer.URI_WSDL11_SOAP11 + "/" + WSDeployer.URI_WSDL11_SOAP12;
    }
    return null;
  }

  /**
   * return null if transport is supported by Axis engine. Else, returns a
   * message explaining the problem.
   *
   * @param soapBinding - soapBinding to check
   * @return null if transport is supported by Axis engine. Else, returns a
   *         message explaining the problem.
   */
  @Override
  protected String checkTransport(final ExtensibilityElement soapBinding) {
    String transportUri = "";
    if (soapBinding instanceof SOAPBinding) {
      transportUri = ((SOAPBinding) soapBinding).getTransportURI();
    } else if (soapBinding instanceof SOAP12Binding) {
      transportUri = ((SOAP12Binding) soapBinding).getTransportURI();
    }
    if (!WSDeployer.SOAP_HTTP_TRANSPORT_URI.equals(transportUri)) {
      return "Transport URI : " + transportUri + " is not supported. Please use " + WSDeployer.SOAP_HTTP_TRANSPORT_URI;
    }
    return null;
  }

  /**
   * return null if use is supported by Axis engine. Else, returns a message
   * explaining the problem.
   *
   * @param soapBody - soapBody to check
   * @return null if use is supported by Axis engine. Else, returns a message
   *         explaining the problem.
   */
  @Override
  protected String checkUse(final ExtensibilityElement element) {
    String use = "";
    if (element instanceof SOAPBody) {
      use = ((SOAPBody) element).getUse();
    } else if (element instanceof SOAP12Body) {
      use = ((SOAP12Body) element).getUse();
    } else if (element instanceof SOAPFault) {
      use = ((SOAPFault) element).getUse();
    } else if (element instanceof SOAP12Fault) {
      use = ((SOAP12Fault) element).getUse();
    }
    if (!"literal".equals(use)) {
      return "Use : " + use + " is not supported. Please use " + "literal";
    }
    return null;
  }

  /**
   * return null if soapBody attributes are supported by Axis engine. Else,
   * returns a message explaining the problem.
   *
   * @param soapBody - soapBody to check
   * @return null if soapBody attributes are supported supported by Axis engine.
   *         Else, returns a message explaining the problem.
   */
  @Override
  protected String checkSoapBody(final ExtensibilityElement soapBody) {
    List parts = null;
    if (soapBody instanceof SOAPBody) {
      parts = ((SOAPBody) soapBody).getParts();
    } else if (soapBody instanceof SOAP12Body) {
      parts = ((SOAP12Body) soapBody).getParts();
    }
    if (parts != null) {
      return "SoapBody is using parts attribute which is not currently supported.";
    }
    return null;
  }

  @Override
  protected String checkSoapFault(final ExtensibilityElement soapFault) {
    String name = null;
    if (soapFault instanceof SOAPFault) {
      name = ((SOAPFault) soapFault).getName();
    } else if (soapFault instanceof SOAP12Fault) {
      name = ((SOAP12Fault) soapFault).getName();
    }
    if (name == null) {
      return "SoapFault is not specifying fault name which is not currently supported.";
    }
    return null;
  }

  /**
   * Check if WS engine is available. Else throw an exception.
   */
  @Override
  protected void checkWSEngineIsAvailable() {
    try {
      ServiceAdmin.getEngine();
    } catch (final Exception e) {
      throw new OrchestraRuntimeException("Axis web container is not started or Axis engine is not deployed", e);
    }

  }

  /**
   * This method will deploy the specified WS Service on the choosen ws engine.
   * This method must be overriden by each ws engine Deployer.
   *
   * @param def - Definition object that represents a WS Service to deploy.
   */
  @Override
  protected void deployServices(final List<Service> services) {
    for (final Service service : services) {
      final File wsdlFile = this.createWsdlFile(service);

      for (final Port port : (Collection<Port>) service.getPorts().values()) {
        final String endpointURL = this.expectedAddressPrefix + this.getServiceName(port);
        final Map<String, Boolean> isOneWay = new HashMap<String, Boolean>();
        final Map<String, Boolean> locks = new HashMap<String, Boolean>();
        this.getLocksAndOneWays(port, locks, isOneWay);
        // register process endpoint if no other version of the process is already deployed
        if (this.axisPublisher.addProcessEndpoint(
                endpointURL, this.bpelProcess.getQName(), port, this.bpelProcess.getUUID(), locks, isOneWay)) {
          // deploy port only when new process endpoint.
          this.deployPort(port, service.getQName(), wsdlFile);
        }
      }
    }
  }

  protected void deployPort(final Port port, final QName serviceQName, final File wsdlFile) {
    final QName javaURI = new QName(WSDDConstants.URI_WSDD_JAVA, WSDDBPELMsgProvider.PROVIDER_NAME);
    WSDDProvider.registerProvider(javaURI, new WSDDBPELMsgProvider());
    this.createServicesDir();

    final String endpointName = this.getServiceName(port);
    try {
      final WSDDDocument wsddDoc = this.createWsddDeployment(port, serviceQName, wsdlFile);

      // update implementations
      final String writtenClass = this.updateImplementationFile(port);
      // compile java classes,
      this.compileClass(writtenClass);
      // add this to process classloader
      this.addClassesToClassLoader(endpointName);

      // load wsdd file into axis webapp
      this.callAdminClientTask(wsddDoc, endpointName);
    } catch (final WSDDException e) {
      throw new OrchestraRuntimeException("error while creating WSDD definition.", e);
    }
  }


  /**
   * This method will undeploy the specified WS Service from the choosen ws
   * engine. This method must be overriden by each ws engine Deployer.
   *
   * @param def - Definition object that represents a WS Service to undeploy.
   */
  @Override
  protected void undeployServices(final List<Service> services) {
    for (final Service service : services) {
      for (final Port port : (Collection<Port>) service.getPorts().values()) {
        final String endpointName = this.getServiceName(port);
        final String endpointURL = this.expectedAddressPrefix + endpointName;
        // stop service only if no other versions of the process use it.
        if (this.axisPublisher.removeProcessEndpoint(endpointURL, this.bpelProcess.getUUID())) {
          this.undeployPort(port);
          this.removeClassLoader(endpointName);
        }
      }
    }
  }

  protected void undeployPort(final Port port) {
    try {
      final WSDDDocument wsddUndeployDoc = this.createWsddUndeployDocument(port);
      this.callAdminClientTask(wsddUndeployDoc, this.getServiceName(port));
      // Do not delete WS classes: may still be locked by classLoader
      //    this.deleteServicesDir();
    } catch (final WSDDException e) {
      throw new OrchestraRuntimeException("error while creating WSDD definition.", e);
    }
  }

  /**
   * Call axis admin client with the given wsdd file to deploy/undeploy
   * services.
   *
   * @param wsddFile - wsdd file to use
   */
  private void callAdminClientTask(final WSDDDocument wsddDoc, final String endpointName) {
    final ClassLoader old = ClassUtils.getDefaultClassLoader();
    try {
      final ClassLoader processClassLoader = ClassLoaderMap.getCL(endpointName);
      ClassUtils.setDefaultClassLoader(processClassLoader);

      final AxisServer engine = ServiceAdmin.getEngine();
      engine.setOption(CommandService.class.getName(), Environment.getFromCurrent(CommandService.class));
      final EngineConfiguration config = engine.getConfig();
      if (config instanceof WSDDEngineConfiguration) {
        final WSDDDeployment deployment = ((WSDDEngineConfiguration)config).getDeployment();
        for (final WSDDService serviceToDeploy : wsddDoc.getDeployment().getServices()) {
          if (deployment.getWSDDService(serviceToDeploy.getQName()) != null) {
            throw new OrchestraRuntimeException("A service has already been deployed for url: "
                + this.expectedAddressPrefix + serviceToDeploy.getQName().getLocalPart());
          }
        }
        wsddDoc.deploy(deployment);
      }
      engine.refreshGlobalOptions();
      engine.saveConfiguration();

    } catch (final OrchestraRuntimeException e) {
      throw e;
    } catch (final Exception e) {
      throw new OrchestraRuntimeException("Exception during Web service deployment.", e);
    } finally {
      ClassUtils.setDefaultClassLoader(old);
    }
  }

  /**
   * Add axis WS classes to process classLoader
   *
   */
  private void addClassesToClassLoader(final String endpointName) {
    final ClassLoader parent = AxisDeployer.class.getClassLoader();
    final ClassLoader envClassLoader = Environment.getCurrent().getClassLoader();
    URL url = null;
    try {
      url = this.servicesDir.toURI().toURL();
    } catch (final MalformedURLException e) {
      throw new OrchestraRuntimeException(e);
    }
    final URL[] urls = new URL[] {
      url
    };
    final WebServiceClassLoader cl = new WebServiceClassLoader(endpointName, urls, parent, envClassLoader);
    ClassLoaderMap.addCL(endpointName, cl);
  }

  private void removeClassLoader(final String endpointName) {
    ClassLoaderMap.removeCL(endpointName);
  }
  /**
   * Compiles java classes from basedir.
   */
  private void compileClass(final String writtenClass) {
    // Create the compilation context
    final CompilationContext context = new CompilationContext();
    context.setContextualClassLoader(this.getClass().getClassLoader());
    context.setOutputDirectory(this.servicesDir);
    context.setSourceDirectory(this.servicesDir);

    // Assign the empty list of sources to be compiled
    context.setSources(Arrays.asList(writtenClass));
    // create the compiler
    final JOnASCompiler compiler = new JOnASCompiler(context);

    final List<CompilerError> errors = compiler.compile();

    final StringBuffer errorMsg = new StringBuffer();
    for (final CompilerError error : errors) {
      errorMsg.append(error.toString()).append(Misc.LINE_SEPARATOR);
    }
    if (errorMsg.length() > 0) {
      throw new OrchestraRuntimeException("Error during compilation: " + errorMsg);
    }
  }

  /**
   * Create new java implementations for each binding of the given definition.
   *
   * @param def - the definition to analyse.
   * @return a list of created classes
   */
  private String updateImplementationFile(final Port port) {
    final Binding binding = port.getBinding();
    return new BindingFileWriter(this.servicesDir, binding, this.bpelProcess).write();
  }

  /**
   * Return the deploy wsdl file of the given definition.
   *
   * @param def : def to analyse
   * @return the deploy wsdl file used by the given def
   */
  private File getWsdlFile(final QName serviceQName) {
    return new File(this.getWsddFilesDir(serviceQName.getNamespaceURI())
        + File.separator + serviceQName.getLocalPart() + AxisDeployer.WSDL_FILE_NAME);
  }

  /**
   * Returns the directory where wsdd files of this definition could be found.
   *
   * @param def : def to analyse
   * @return the directory where wsdd files of this definition could be found
   */
  private String getWsddFilesDir(final String wsdlTargetnamespace) {
    final String wsddFilesDir =
      this.servicesDir
      + File.separator
      + AxisDeployer.getDirectoryFromPackage(
          AxisDeployer.getPackageFromNamespace(wsdlTargetnamespace));
    return wsddFilesDir;
  }

  /*
   * UTILITY METHODS
   */

  /**
   * Set basedir field.
   */
  private void createServicesDir() {
    try {
      if (this.servicesDir.isDirectory()) {
        return;
      }
      if (!this.servicesDir.mkdir()) {
        throw new IOException("Cannot create the directory '" + this.servicesDir + "'.");
      }
      if (AxisDeployer.log.isLoggable(Level.FINE)) {
        AxisDeployer.log.fine("Services directory created : " + this.servicesDir.getAbsolutePath());
      }
    } catch (final Exception e) {
      throw new OrchestraRuntimeException("Error creating " + this.servicesDir, e);
    }
  }

  private static synchronized File getWStempDir() {
    if (AxisDeployer.webAppTmpDir == null) {
      try {
        new File(System.getProperty("java.io.tmpdir")).mkdirs();
        AxisDeployer.webAppTmpDir = File.createTempFile("orchWS-Repo-", null, null);
        AxisDeployer.webAppTmpDir.delete();
        if (!AxisDeployer.webAppTmpDir.mkdirs()) {
          throw new IOException("Cannot create the temporary directory '" + AxisDeployer.webAppTmpDir + "'.");
        }
        if (AxisDeployer.log.isLoggable(Level.FINE)) {
          AxisDeployer.log.fine("webAppTmpDir created : " + AxisDeployer.webAppTmpDir.getAbsolutePath());
        }
      } catch (final Exception e) {
        throw new OrchestraRuntimeException("Error creating " + AxisDeployer.webAppTmpDir, e);
      }
    }
    return AxisDeployer.webAppTmpDir;
  }

  synchronized void resetServiceDir() {
    if (this.servicesDir != null) {
      Misc.deleteDir(this.servicesDir);
    }
  }

  private File getServicesDir() {
    final String packag = Utils.makePackageName(this.bpelProcess.getTargetNamespace());
    return new File(AxisDeployer.getWStempDir() + File.separator + packag + "__" + this.bpelProcess.getName());
  }

  /**
   * Returns the java package that maps to the given namespace.
   *
   * @param ns - ns
   * @return the java package that maps to the given namespace.
   */
  public static String getPackageFromNamespace(final String ns) {
    final String packag = Utils.makePackageName(ns);
    return packag;
  }

  /**
   * Returns the directory structure corresponding to the given package.
   *
   * @param packag - packag
   * @return the directory structure corresponding to the given package.
   */
  public static String getDirectoryFromPackage(final String packag) {
    final String dir = packag.replace('.', File.separatorChar);
    return dir;
  }

  private File createWsdlFile(final Definition wsdlDefinition, final File wsdlDir) {
    try {
      final Map<String, List<Import>> imports = wsdlDefinition.getImports();
      if (!imports.isEmpty()) {
        for (final List<Import> l : imports.values()) {
          for (final Import impor : l) {
            final Definition importedDefinition = impor.getDefinition();
            final File importFile = this.createWsdlFile(importedDefinition, wsdlDir);
            impor.setLocationURI(importFile.getName());
          }
        }
      }
      final File wsdlFile = File.createTempFile("generated", ".wsdl", wsdlDir);
      final FileOutputStream fos = new FileOutputStream(wsdlFile);
      try {
        WsdlUtil.writeWsdl(wsdlDefinition, fos);
      } finally {
        fos.close();
      }
      return wsdlFile;
    } catch (final IOException e) {
      throw new OrchestraRuntimeException(e);
    }
  }

  private File createWsdlFile(final Service service) {
    final File wsdlFile = this.getWsdlFile(service.getQName());
    final File wsdlDir = wsdlFile.getParentFile();
    wsdlDir.mkdirs();

    final Definition wsdlDefinition = this.bpelProcess.getWsdlInfos().extractServiceWsdlDefinition(service);
    return this.createWsdlFile(wsdlDefinition, wsdlDir);
  }

  private WSDDDocument createWsddDeployment(final Port port, final QName serviceQName, final File wsdlFile) throws WSDDException {
    final StringWriter wsddWriter = new StringWriter();

    wsddWriter.write("<?xml version='1.0' encoding='UTF-8'?>\n");
    wsddWriter.write("<deployment xmlns='http://xml.apache.org/axis/wsdd/' "
        + "    xmlns:java='http://xml.apache.org/axis/wsdd/providers/java'>\n");

    final QName bindingQN = port.getBinding().getQName();

    String className = bindingQN.getLocalPart() + "Impl";
    final String firstLetter = className.substring(0, 1);
    final String nextLetters = className.substring(1);
    className = firstLetter.toUpperCase() + nextLetters;
    final String serviceName = this.getServiceName(port);

    wsddWriter.write("<service name='" + serviceName + "' provider='java:BPELMsg' style='Message' use='literal'>\n"
        + "<parameter name='wsdlTargetNamespace' value='" + serviceQName.getNamespaceURI() + "'/>\n"
        + "<parameter name='wsdlServiceElement' value='" + serviceQName.getLocalPart() + "'/>\n"
        + "<parameter name='wsdlServicePort' value='" + port.getName() + "'/>\n"
        + "<parameter name='className' value='"
        + AxisDeployer.getPackageFromNamespace(bindingQN.getNamespaceURI()) + "." + className + "'/>\n"
        + "<parameter name='wsdlPortType' value='" + port.getBinding().getPortType().getQName().getLocalPart() + "'/>\n"
        + "<parameter name='typeMappingVersion' value='1.3'/>\n"
        + "<parameter name='scope' value='Request'/>\n");
    for (final Operation operation : (Collection<Operation>) port.getBinding().getPortType().getOperations()) {
      wsddWriter.write(
          "<operation name='" + operation.getName()
          + "' qname='operNs:" + operation.getName()
          + "' xmlns:operNs='" + port.getBinding().getPortType().getQName().getNamespaceURI() + "' ");
      if (operation.getStyle().equals(OperationType.ONE_WAY)) {
        wsddWriter.write("mep='oneway' />\n");
      } else {
        // TODO: add operation parameters and return type description
        wsddWriter.write(" />\n");

      }
    }
    wsddWriter.write("  <wsdlFile>" + wsdlFile.getAbsolutePath() + "</wsdlFile>\n");
    wsddWriter.write("  <endpointURL>" + this.expectedAddressPrefix + "</endpointURL>\n");
    wsddWriter.write("</service>\n");
    wsddWriter.write("</deployment>");

    return new WSDDDocument(XmlUtil.getDocumentFromString(wsddWriter.toString()));
  }

  private String getServiceName(final Port port) {
    final String endpointURL = this.getAddressFromPort(port);
    String serviceName = port.getName();
    if (endpointURL != null) {
      serviceName = endpointURL.substring(this.expectedAddressPrefix.length());
    }
    return serviceName;
  }

  private WSDDDocument createWsddUndeployDocument(final Port port) throws WSDDException {
    final StringWriter wsddWriter = new StringWriter();

    wsddWriter.write("<?xml version='1.0' encoding='UTF-8'?>\n");
    wsddWriter.write("<undeployment xmlns='http://xml.apache.org/axis/wsdd/'>\n");
    wsddWriter.write("<service name='" + this.getServiceName(port) + "'/>\n");
    wsddWriter.write("</undeployment>");

    return new WSDDDocument(XmlUtil.getDocumentFromString(wsddWriter.toString()));
  }
}
