/**
 * Copyright (C) 2010  Bull S. A. S.
 * Bull, Rue Jean Jaures, B.P.68, 78340, Les Clayes-sous-Bois
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation
 * version 2.1 of the License.
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA  02110-1301, USA.
 **/
package org.ow2.orchestra.cluster.jgroups;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import java.util.logging.Level;

import org.jgroups.Address;
import org.jgroups.ExtendedReceiverAdapter;
import org.jgroups.JChannel;
import org.jgroups.Message;
import org.jgroups.View;
import org.jgroups.util.Util;
import org.ow2.orchestra.cluster.ClusterDescription;
import org.ow2.orchestra.cluster.JmxServer;
import org.ow2.orchestra.cluster.Server;
import org.ow2.orchestra.facade.exception.OrchestraRuntimeException;
import org.ow2.orchestra.util.Misc;
import org.ow2.orchestra.util.OrchestraConstants;


/**
 * Orchestra cluster implementation using JGroups.
 *
 * To use this cluster implementation, add this to the environment.xml file:
 *
 * <pre>
 * {@code
 * <cluster class='org.ow2.orchestra.cluster.jgroups.JGroupsClusterDescription' init='eager'>
 *   <arg><ref object='orchestra-properties' /></arg>
 *   <arg><string value='jgroups-configuration-url' /></arg>
 * </cluster>
 * }
 * </pre>
 *
 * Where <code>jgroups-configuration-url</code> is the url of the file containing the JGroups configuration.
 *
 * To use the default JGroups configuration, remove the second {@code <arg ...>} line.
 *
 * @author Guillaume Porcher
 *
 */
public class JGroupsClusterDescription implements ClusterDescription {

  private final JChannel channel;
  private final Map<Address, Server> servers = new HashMap<Address, Server>();

  public JGroupsClusterDescription(final Properties orchestraProperties) {
    this(orchestraProperties, null);
  }

  public JGroupsClusterDescription(final Properties orchestraProperties, final String jGroupsConfiguration) {
    // Orchestra properties
    final String jmxObjectName = orchestraProperties.getProperty(OrchestraConstants.JMX_OBJECT_NAME);
    String jmxServiceUrl = orchestraProperties.getProperty(OrchestraConstants.JMX_SERVICE_URL);

    try {
      // replace localhost with host ip
      jmxServiceUrl = jmxServiceUrl.replace("localhost", InetAddress.getLocalHost().getHostAddress());
    } catch (final UnknownHostException e) {
      // keep localhost
    }
    final JmxServer localServer = new JmxServer(jmxServiceUrl, jmxObjectName);

    try {
      this.channel = new JChannel(jGroupsConfiguration);
      this.channel.setReceiver(new ExtendedReceiverAdapter() {
        @Override
        public void receive(final Message msg) {
          // received message
          final Address src = msg.getSrc();
          final Object obj;
          try {
            obj = Util.objectFromByteBuffer(msg.getBuffer());
          } catch (final Exception e) {
            // invalid message
            Misc.log(Level.SEVERE, "received invalid message %s", msg);
            return;
          }
          if (obj instanceof Server) {
            JGroupsClusterDescription.this.servers.put(src, (Server) obj);
            Misc.log(Level.INFO, "received message %s -> %s", msg, obj);
          } else {
            // invalid message
            Misc.log(Level.SEVERE, "received invalid message %s", msg);
            return;
          }
          if (msg.getDest() == null) {
            // hello message: reply with local configuration
            try {
              final Message replyMsg = new Message();
              replyMsg.setBuffer(Util.objectToByteBuffer(localServer));
              replyMsg.setDest(src);
              JGroupsClusterDescription.this.channel.send(replyMsg);
            } catch (final Exception e) {
              Misc.log(Level.SEVERE, "exception while sending reply");
            }
          }

        }
        @Override
        /**
         * Cluster configuration changed: keep only online servers
         */
        public void viewAccepted(final View view) {
          Misc.log(Level.FINE, "Cluster refreshed: %s", view);
          JGroupsClusterDescription.this.servers.keySet().retainAll(view.getMembers());
        }
      });
      this.channel.connect("orchestra-cluster");

      final Message msg = new Message();
      msg.setBuffer(Util.objectToByteBuffer(localServer));
      this.channel.send(msg);
    } catch (final Exception e) {
      throw new OrchestraRuntimeException(e);
    }
  }

  public Collection<Server> getOrchestraServers() {
    return this.servers.values();
  }
}
