package org.ow2.orchestra.osgi;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.Dictionary;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Map;
import java.util.Properties;
import java.util.logging.Level;

import javax.servlet.Servlet;

import org.osgi.framework.BundleContext;
import org.osgi.service.http.HttpContext;
import org.osgi.service.http.HttpService;
import org.ow2.orchestra.env.BpelEnvironmentParser;
import org.ow2.orchestra.jmx.JMXAgent;
import org.ow2.orchestra.pvm.env.EnvironmentFactory;
import org.ow2.orchestra.services.itf.HttpPublisher;
import org.ow2.orchestra.util.Misc;
import org.ow2.orchestra.util.OrchestraConstants;
import org.ow2.orchestra.util.XmlUtil;

public class OrchestraOSGiEngine {

  public static final String ORCHESTRA_WAR_CONTEXT_PROP = "orchestra.war.context";

  private final Map<String, Servlet> aliases;

  // Injected by iPOJO
  private HttpService httpService = null;
  private OrchestraExtensionService orchestraExtensionService = null;
  private URL environmentLocation = null;


  private JMXAgent jmxAgent;
  private EnvironmentFactory environmentFactory;

  private ClassLoader bundleClassLoader;
  private final BundleContext bundleContext;

  public OrchestraOSGiEngine(final BundleContext bundleContext) {
    this.aliases = new HashMap<String, Servlet>();
    this.bundleContext = bundleContext;
  }


  public void setEnvironmentLocation(final String environmentLocation) throws MalformedURLException {
    this.environmentLocation = new URL(environmentLocation);
  }
  public URL getEnvironmentLocation() {
    return this.environmentLocation;
  }
  public HttpService getHttpService() {
    return this.httpService;
  }

  public void setHttpService(final HttpService httpService) {
    this.httpService = httpService;
  }

  public void unsetHttpService(final HttpService httpService) {
    // unregister servlets
    for (final String alias : this.aliases.keySet()) {
      this.httpService.unregister(alias);
      Misc.log(Level.INFO, "Alias unregistered: %s", alias);
    }

    this.httpService = null;

  }


  /**
   * @param orchestraExtensionService the orchestraExtensionService to set
   */
  public void setOrchestraExtensionService(final OrchestraExtensionService orchestraExtensionService) {
    this.orchestraExtensionService = orchestraExtensionService;
    this.bundleClassLoader = this.orchestraExtensionService.getClassLoader();
  }

  public void unsetOrchestraExtensionService(final OrchestraExtensionService orchestraExtensionService) {
    this.orchestraExtensionService = null;
    this.bundleClassLoader = null;
  }


  public void start() throws Throwable {
    final ClassLoader originalClassLoader = Thread.currentThread().getContextClassLoader();
    Thread.currentThread().setContextClassLoader(this.bundleClassLoader);
    try {
      Misc.log(Level.FINEST, "%s starting", this.getClass().getName());

      // open environment
      Misc.log(Level.INFO, "Environment will be taken from: %s", this.environmentLocation);

      this.environmentFactory = BpelEnvironmentParser.parseEnvironmentFactoryFromURL(this.environmentLocation);

      // register cxfServlet
      // get web app path from orchestra.properties
      final Properties orchestraProperties =
        (Properties) this.environmentFactory.get(OrchestraConstants.ORCHESTRA_PROPERTIES);
      final String servletPath = orchestraProperties.getProperty(OrchestraConstants.SERVLET_PATH);

      final HttpContext context = this.httpService.createDefaultHttpContext();
      Misc.log(Level.FINEST, "Context created: " + context);

      final HttpPublisher publisher = this.environmentFactory.get(HttpPublisher.class);
      final Servlet servlet = publisher.getPublisherServlet();
      Misc.log(Level.FINEST, "Servlet created: %s", servlet);
      final Dictionary<String, String> initparams = new Hashtable<String, String>();


      final String inWar = this.bundleContext.getProperty(OrchestraOSGiEngine.ORCHESTRA_WAR_CONTEXT_PROP);
      if (inWar != null && Boolean.TRUE.equals(XmlUtil.parseBooleanValue(inWar))) {
        this.aliases.put("/", servlet);
      } else {
        this.aliases.put("/" + servletPath, servlet);
      }

      for (final Map.Entry<String, Servlet> mapping : this.aliases.entrySet()) {
        this.httpService.registerServlet(mapping.getKey(), mapping.getValue(), initparams, context);
        Misc.log(Level.INFO, "Alias registered: %s for servlet: %s", mapping.getKey(), mapping.getValue());
      }

      // create jmxAgent
      this.jmxAgent = new JMXAgent();

      // start orchestra
      this.jmxAgent.startOrchestra(this.environmentFactory);
    } catch (final Throwable t) {
      t.printStackTrace();
      throw t;
    } finally {
      Thread.currentThread().setContextClassLoader(originalClassLoader);
    }
  }

  public void stop() throws Throwable {
    final ClassLoader originalClassLoader = Thread.currentThread().getContextClassLoader();
    if (this.bundleClassLoader != null) {
      Thread.currentThread().setContextClassLoader(this.bundleClassLoader);
    }
    try {
      Misc.log(Level.FINEST, "%s stopping", this.getClass().getName());


      // stop orchestra
      if (this.jmxAgent != null) {
        this.jmxAgent.stopOrchestra();
      }
    } catch (final Throwable t) {
      t.printStackTrace();
      throw t;
    } finally {
      Thread.currentThread().setContextClassLoader(originalClassLoader);
    }
  }
}
