/**
 * Copyright (C) 2009  Bull S. A. S.
 * Bull, Rue Jean Jaures, B.P.68, 78340, Les Clayes-sous-Bois
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation
 * version 2.1 of the License.
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA  02110-1301, USA.
 **/
package org.ow2.orchestra;


/**
 * @author Guillaume Porcher
 *
 */
import org.apache.felix.framework.Felix;
import org.apache.felix.framework.util.Util;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleContext;
import org.osgi.framework.Constants;
import org.osgi.framework.FrameworkEvent;
import org.ow2.orchestra.osgi.OrchestraOSGiEngine;

import javax.servlet.ServletContext;
import javax.servlet.ServletContextEvent;
import javax.servlet.ServletContextListener;
import java.io.InputStream;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;
import java.util.logging.Level;
import java.util.logging.Logger;

public final class StartupListener implements ServletContextListener {

  private static final Logger LOG = Logger.getLogger(StartupListener.class.getName());
  private Felix felix;

  public void contextInitialized(final ServletContextEvent event) {
    try {

      final InputStream inStream = StartupListener.class.getClassLoader().getResourceAsStream("conf/config.properties");
      final Properties props = new Properties();
      try {
        props.load(inStream);
      } finally {
        inStream.close();
      }

      // Perform variable substitution
      for (final Object key : props.keySet()) {
        final String property = (String) key;
        props.setProperty(property, Util.substVars(props.getProperty(property), property, null, props));
      }

      if (!props.containsKey("felix.cache.rootdir")) {
        props.put("felix.cache.rootdir", System.getProperty("java.io.tmpdir"));
      }

      // add property to tell orchestra we are running from a war
      props.put(OrchestraOSGiEngine.ORCHESTRA_WAR_CONTEXT_PROP, event.getServletContext().getContextPath());


      this.felix = new Felix(props);
      this.felix.init();
      // Start the framework.
      this.felix.start();
      // Use the system bundle context to process the auto-deploy
      // and auto-install/auto-start properties.
      this.installBundles(this.felix.getBundleContext(), event.getServletContext());

      event.getServletContext().setAttribute(BundleContext.class.getName(), this.felix.getBundleContext());
      StartupListener.LOG.info("OSGi framework started");
    } catch (final Exception e) {
      StartupListener.LOG.log(Level.SEVERE, "Error starting framework", e);
    }
  }

  public void contextDestroyed(final ServletContextEvent event) {
    try {
      if (this.felix != null) {
        this.felix.stop();
        final FrameworkEvent fe = this.felix.waitForStop(60000);
        if (FrameworkEvent.STOPPED == fe.getType()) {
          StartupListener.LOG.info("Framework stopped.");
        } else if (fe.getThrowable() != null) {
          StartupListener.LOG.log(Level.SEVERE, "Error stopping framework", fe.getThrowable());
        } else {
          StartupListener.LOG.warning("Framework stopped in state " + fe.getType() + ".");
        }
        this.felix = null;
      }
    } catch (final Exception e) {
      StartupListener.LOG.log(Level.SEVERE, "Error stopping framework", e);
    }
  }

  public void installBundles(final BundleContext context, final ServletContext servletContext) throws Exception {

    final ArrayList<Bundle> installed = new ArrayList<Bundle>();
    for (final URL url : this.findBundles(servletContext)) {
      StartupListener.LOG.log(Level.FINE, "Installing bundle [" + url + "]");
      final Bundle bundle = context.installBundle(url.toExternalForm(), url.openStream());
      installed.add(bundle);
    }


    for (final Bundle bundle : installed) {
      if (bundle.getHeaders().get(Constants.FRAGMENT_HOST) == null) {
        bundle.start();
      }
    }
  }


  private List<URL> findBundles(final ServletContext servletContext) throws Exception {
    final ArrayList<URL> list = new ArrayList<URL>();
    for (final Object o : servletContext.getResourcePaths("/WEB-INF/bundle/")) {
      final String name = (String) o;
      if (name.endsWith(".jar")) {
        final URL url = servletContext.getResource(name);
        if (url != null) {
          list.add(url);
        }
      }
    }

    return list;
  }
}
