package org.ow2.orchestra.cxf.osgi;

import java.net.URL;
import java.util.Dictionary;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Map;
import java.util.Properties;
import java.util.logging.Level;

import javax.servlet.Servlet;
import javax.servlet.http.HttpServlet;

import org.osgi.service.http.HttpContext;
import org.osgi.service.http.HttpService;
import org.ow2.orchestra.cxf.OrchestraBaseCxfServlet;
import org.ow2.orchestra.env.BpelEnvironmentParser;
import org.ow2.orchestra.jmx.JMXAgent;
import org.ow2.orchestra.pvm.env.EnvironmentFactory;
import org.ow2.orchestra.util.Misc;
import org.ow2.orchestra.util.OrchestraConstants;

public class Engine {

  private final Map<String, Servlet> aliases;

  // Injected by iPOJO
  private HttpService httpService = null;
  private String environmentLocation = null;

  private JMXAgent jmxAgent;

  public Engine() {
    this.aliases = new HashMap<String, Servlet>();
  }

  public void setEnvironment(final String environmentLocation) {
    this.environmentLocation = environmentLocation;
  }

  public String getEnvironment() {
    return this.environmentLocation;
  }

  public HttpService getHttpService() {
    return this.httpService;
  }

  public void setHttpService(final HttpService httpService) {
    this.httpService = httpService;
  }

  public void start() throws Throwable {
    final ClassLoader bundleClassLoader = this.getClass().getClassLoader();
    final ClassLoader originalClassLoader = Thread.currentThread().getContextClassLoader();
    Thread.currentThread().setContextClassLoader(bundleClassLoader);
    try {
      Misc.log(Level.FINEST, "%s starting", this.getClass().getName());


      // open environment
      if (this.environmentLocation == null) {
        Misc.log(Level.WARNING, "Can't get the environmentLocation from OSGi "
            + "config, using internal environment. This may not be what you are expecting."
            + " If this is the case, please read your OSGi implementation documentation "
            + "about ConfigAdmin service.");

        Misc.unreachableStatement("Implement the default behaviour which take the environment from the bundle.");
      }
      Misc.log(Level.INFO, "Environment will be taken from: %s", this.environmentLocation);
      final URL url = new URL(this.environmentLocation);

      final EnvironmentFactory envFactory = BpelEnvironmentParser.parseEnvironmentFactoryFromURL(url);

      // register cxfServlet
      // get web app path from orchestra.properties
      final Properties orchestraProperties = (Properties) envFactory.get(OrchestraConstants.ORCHESTRA_PROPERTIES);
      final String servletPath = orchestraProperties.getProperty(OrchestraConstants.SERVLET_PATH);

      final HttpContext context = this.httpService.createDefaultHttpContext();
      Misc.log(Level.FINEST, "Context created: " + context);
      this.httpService.registerResources("/", "index.html", context);
      final HttpServlet servlet = new OrchestraBaseCxfServlet();
      Misc.log(Level.FINEST, "Servlet created: %s", servlet);
      final Dictionary<String, String> initparams = new Hashtable<String, String>();

      this.aliases.put("/" + servletPath, servlet);
      for (final Map.Entry<String, Servlet> mapping : this.aliases.entrySet()) {
        this.httpService.registerServlet(mapping.getKey(), mapping.getValue(), initparams, context);
        Misc.log(Level.INFO, "Alias registered: %s for servlet: %s", mapping.getKey(), mapping.getValue());
      }

      // create jmxAgent
      this.jmxAgent = new JMXAgent();

      // start orchestra
      this.jmxAgent.startOrchestra(envFactory);
    } catch (final Throwable t) {
      t.printStackTrace();
      throw t;
    } finally {
      Thread.currentThread().setContextClassLoader(originalClassLoader);
    }
  }

  public void stop() throws Throwable {
    final ClassLoader bundleClassLoader = this.getClass().getClassLoader();
    final ClassLoader originalClassLoader = Thread.currentThread().getContextClassLoader();
    Thread.currentThread().setContextClassLoader(bundleClassLoader);
    try {
      Misc.log(Level.FINEST, "%s stopping", this.getClass().getName());

      // unregister axis
      for (final String alias : this.aliases.keySet()) {
        this.httpService.unregister(alias);
        Misc.log(Level.INFO, "Alias unregistered: %s", alias);
      }
      this.httpService.unregister("/");
      // stop orchestra
      if (this.jmxAgent != null) {
        this.jmxAgent.stopOrchestra();
      }
    } catch (final Throwable t) {
      t.printStackTrace();
      throw t;
    } finally {
      Thread.currentThread().setContextClassLoader(originalClassLoader);
    }
  }
}
