/**
 * Copyright (C) 2007  Bull S. A. S.
 * Bull, Rue Jean Jaures, B.P.68, 78340, Les Clayes-sous-Bois
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation
 * version 2.1 of the License.
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA  02110-1301, USA.
 **/
package org.ow2.orchestra.cxf.example.loanApprovalCamel;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.xml.namespace.QName;
import javax.xml.soap.SOAPBodyElement;
import javax.xml.soap.SOAPMessage;

import junit.framework.Assert;
import junit.framework.AssertionFailedError;

import org.ow2.orchestra.facade.exception.OrchestraRuntimeException;
import org.ow2.orchestra.test.remote.perf.RemotePerfTestCase;
import org.ow2.orchestra.util.Misc;
import org.ow2.orchestra.util.SOAPUtil;
import org.ow2.orchestra.util.XmlUtil;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.w3c.dom.Text;

/**
 * @author Charles Souillard
 */
public class CamelLoanApprovalTest extends RemotePerfTestCase {

  @Override
  protected String getProcessName() {
    return "loanService";
  }
  @Override
  protected String getProcessNamespace() {
    return "http://orchestra.ow2.org/loanApproval/loanService";
  }



  public void deploy() {
    try {
      final Map<String, byte[]> resources = new HashMap<String, byte[]>();
      resources.put("camel-context.xml", Misc.getAllContentFrom(this.getClass().getResource("camel-context.xml")));

      resources.put(ApprovalBindingImpl.class.getName().replace(".", "/") + ".class",
          Misc.getAllContentFrom(this.getClass().getResource(ApprovalBindingImpl.class.getSimpleName() + ".class")));
      resources.put(ApprovalPT.class.getName().replace(".", "/") + ".class",
          Misc.getAllContentFrom(this.getClass().getResource(ApprovalPT.class.getSimpleName() + ".class")));
      resources.put(RiskAssessmentBindingImpl.class.getName().replace(".", "/") + ".class",
          Misc.getAllContentFrom(this.getClass().getResource(RiskAssessmentBindingImpl.class.getSimpleName() + ".class")));
      resources.put(LoanApprovalRouteBuilder.class.getName().replace(".", "/") + ".class",
          Misc.getAllContentFrom(this.getClass().getResource(LoanApprovalRouteBuilder.class.getSimpleName() + ".class")));
      resources.put("approval.wsdl", Misc.getAllContentFrom(this.getClass().getResource("approval.wsdl")));
      resources.put("riskAssessment.wsdl", Misc.getAllContentFrom(this.getClass().getResource("riskAssessment.wsdl")));
      resources.put("common.wsdl", Misc.getAllContentFrom(this.getClass().getResource("common.wsdl")));

      this.deploy(CamelLoanApprovalTest.class.getResource(this.getProcessName() + ".bpel"), null, resources);
    } catch (final Exception e) {
      throw new OrchestraRuntimeException(e);
    }

  }

  @Override
  public void undeploy() {

    this.undeploy(new QName(this.getProcessNamespace(), this.getProcessName()));

  }

  public long launch() {
    long endTime = -1;
    final int amount = Misc.random(0, 60000);
    final SOAPMessage response = SOAPUtil.call(this.getRequest("firstName", "Dupont", Integer.toString(amount)), this.getDefaultEndpoint("loanServicePort"));

    endTime = System.currentTimeMillis();
    if (amount >= 30000) {
      this.assertResult(response, "no");
    } else {
      this.assertResult(response, "yes");
    }
    return endTime;
  }

  public void testLoanApproval() {
    try {
      this.deploy();
      final String endpoint = this.getDefaultEndpoint("loanServicePort");

      // amount < 10000 => assess => risk=low => yes
      this.assertResult(SOAPUtil.call(this.getRequest("firstName", "Hash", "9999"),
          endpoint), "yes");
      // amount < 10000 => assess => risk!=low => approve => risk = medium &&
      // (10000<30000) => yes
      this.assertResult(SOAPUtil.call(this.getRequest("firstName", "Dupont", "9999"),
          endpoint), "yes");
      // amount < 10000 => assess => risk!=low => approve => risk = high &&
      // (9999<10000) => yes
      this.assertResult(SOAPUtil.call(this.getRequest("firstName", "Martin", "9999"),
          endpoint), "yes");
      // amount >= 10000 => approve => amount<50000 => yes
      this.assertResult(SOAPUtil.call(this.getRequest("firstName", "Hash", "10000"),
          endpoint), "yes");
      // amount >= 10000 => approve => amount>=50000 => no
      this.assertResult(SOAPUtil.call(this.getRequest("firstName", "Hash", "50000"),
          endpoint), "no");
      // amount >= 10000 => approve => risk = medium && amount>=30000 => no
      this.assertResult(SOAPUtil.call(this.getRequest("firstName", "Dupont", "30000"),
          endpoint), "no");
      // amount >= 10000 => approve => risk = medium && amount <30000 => yes
      this.assertResult(SOAPUtil.call(this.getRequest("firstName", "Dupont", "20000"),
          endpoint), "yes");
      // amount >= 10000 => approve => risk = high && amount>=10000 => no
      this.assertResult(SOAPUtil.call(this.getRequest("firstName", "Martin", "10000"),
          endpoint), "no");


      this.undeploy();
    } catch (final Exception e) {
      throw new OrchestraRuntimeException(e);
    }
  }

  private void assertResult(final SOAPMessage response, final String expected) {
    if (response != null) {
      SOAPBodyElement operationElement = null;
      try {
        operationElement = this.getSOAPBodyElement(response);

        Assert.assertNotNull(operationElement);
//        Assert.assertEquals(XmlUtil.toString(operationElement), "request", operationElement.getLocalName());
        final List<Element> elements = XmlUtil.elements(operationElement);
        Assert.assertEquals(1, elements.size());
        final Element partElement = elements.get(0);
        Assert.assertEquals("accept", partElement.getLocalName());
        final NodeList texts = partElement.getChildNodes();
        Assert.assertEquals(1, texts.getLength());
        Assert.assertTrue(texts.item(0) instanceof Text);
        Assert.assertEquals(expected, texts.item(0).getTextContent());
      } catch (final Exception e) {
        throw new OrchestraRuntimeException(e);
      } catch (final AssertionFailedError e) {
        System.err.println("AssertionFailed, operationElement = "
            + XmlUtil.toString(operationElement));
        throw e;
      }
    } else {
      Assert.assertTrue("No return available", false);
    }
  }

  private SOAPMessage getRequest(final String firstName, final String name, final String amount) {
    return SOAPUtil.buildDocumentSOAPMessage(
        "http://orchestra.ow2.org/loanApproval/loanService",
        "<loan:request xmlns:loan='http://orchestra.ow2.org/loanApproval/common'>"
            + "<firstName>" + firstName + "</firstName>" + "<name>" + name
            + "</name>" + "<amount>" + amount + "</amount>" + "</loan:request>");
  }

  public String getAlias() {
    return "axis.loan";
  }
}
