/**
 * Copyright (C) 2009  Bull S. A. S.
 * Bull, Rue Jean Jaures, B.P.68, 78340, Les Clayes-sous-Bois
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation
 * version 2.1 of the License.
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA  02110-1301, USA.
 **/
package org.ow2.orchestra.cxf.test.wsaddressing;

import javax.xml.namespace.QName;
import javax.xml.soap.SOAPBodyElement;
import javax.xml.soap.SOAPElement;
import javax.xml.soap.SOAPFault;
import javax.xml.soap.SOAPMessage;

import junit.framework.Assert;

import org.ow2.orchestra.facade.exception.OrchestraRuntimeException;
import org.ow2.orchestra.test.remote.RemoteTestCase;
import org.ow2.orchestra.util.SOAPUtil;
import org.ow2.orchestra.util.XmlConstants;
import org.ow2.orchestra.util.XmlUtil;
import org.w3c.dom.Element;
import org.w3c.dom.Text;

/**
 * @author Guillaume Porcher
 */
public class WSATest extends RemoteTestCase {
  private final String helloProcessName = "helloworld";
  private final String helloProcessNamespace = "http://orchestra.ow2.org/cxf/test/wsaddressing";

  private final String invokerProcessName = "invoker";
  private final String invokerProcessNamespace = "http://orchestra.ow2.org/cxf/test/wsaddressing/invoker";
  /**
   * test helloWorld: process configured to require WS-Addressing.
   * Check receive activities work with WS-Addressing
   */
  public void testHelloworldWithAddressing() {
    try {
      final QName processQName = new QName(this.helloProcessNamespace, this.helloProcessName);

      this.deploy(this.helloProcessName);


      /*
       * First call: no addressing action in header -> fail
       */
      String part = "<submit xmlns='http://orchestra.ow2.org/cxf/test/wsaddressing'><st>Dupont</st></submit>";
      SOAPMessage request = SOAPUtil.buildDocumentSOAPMessage(
          "http://orchestra.ow2.org/cxf/test/wsaddressing/helloworldPT/submitRequest", part);

      final String endpoint = this.getDefaultEndpoint("helloworldPort");

      SOAPMessage response = SOAPUtil.call(request, endpoint);
      if (response != null) {
        final SOAPBodyElement result = this.getSOAPBodyElement(response);
        Assert.assertTrue(result instanceof SOAPFault);
      } else {
        Assert.assertTrue("No return available", false);
      }

      /*
       * Second call: addressing action in header -> success
       */
      part = "<submit xmlns='http://orchestra.ow2.org/cxf/test/wsaddressing'><st>Toto</st></submit>";
      request = SOAPUtil.buildDocumentSOAPMessage(
          "http://orchestra.ow2.org/cxf/test/wsaddressing/helloworldPT/submitRequest", part);
      request.getSOAPHeader().addNamespaceDeclaration("wsa", XmlConstants.ADDRESSING_2005_REF_NAMESPACE);
      final SOAPElement wsaAction = request.getSOAPHeader().addChildElement("Action", "wsa");
      wsaAction.addTextNode("http://orchestra.ow2.org/cxf/test/wsaddressing/helloworldPT/submitRequest");
      request.saveChanges();
      response = SOAPUtil.call(request, endpoint);
      if (response != null) {
        final SOAPBodyElement result = (SOAPBodyElement) response.getSOAPBody()
            .getChildElements().next();
        this.checkReturn(result, "Toto");
      } else {
        Assert.assertTrue("No return available", false);
      }
      this.undeploy(processQName);
    } catch (final Exception e) {
      throw new OrchestraRuntimeException(e);
    }
  }

  /**
   * test invoker: process configured to invoke helloworld with WS-Addressing.
   * Check invoke activities work with WS-Addressing.
   */
  public void testInvokerWithAddressing() {
    try {
      final QName helloProcessQName = new QName(this.helloProcessNamespace, this.helloProcessName);
      final QName invokerProcessQName = new QName(this.invokerProcessNamespace, this.invokerProcessName);

      this.deploy(this.helloProcessName);
      this.deploy(this.invokerProcessName);


      /*
       * First call: no addressing action in header -> ok
       */
      final String part = "<submit xmlns='http://orchestra.ow2.org/cxf/test/wsaddressing'><st>Dupont</st></submit>";
      final SOAPMessage request = SOAPUtil.buildDocumentSOAPMessage(
          "http://orchestra.ow2.org/cxf/test/wsaddressing/invoker/invokerPT/submitRequest", part);

      final String endpoint = this.getDefaultEndpoint("invokerPort");

      final SOAPMessage response = SOAPUtil.call(request, endpoint);
      if (response != null) {
        final SOAPBodyElement result = this.getSOAPBodyElement(response);
        this.checkReturn(result, "Dupont");
      } else {
        Assert.assertTrue("No return available", false);
      }

      this.undeploy(invokerProcessQName);
      this.undeploy(helloProcessQName);
    } catch (final Exception e) {
      throw new OrchestraRuntimeException(e);
    }
  }

  private void checkReturn(final SOAPBodyElement result, final String name) {
    Assert.assertNotNull(result);
    Assert.assertEquals(XmlUtil.toString(result), "submit", result.getElementName().getLocalName());
    Assert.assertEquals(XmlUtil.toString(result), this.helloProcessNamespace, result.getElementName().getURI());

    Assert.assertNotNull(XmlUtil.toString(result), result.getFirstChild());
    Assert.assertNotNull(XmlUtil.element(result));
    final Element st = XmlUtil.element(result);
    Assert.assertEquals("st", st.getLocalName());
    Assert.assertEquals(XmlUtil.toString(st), this.helloProcessNamespace, st.getNamespaceURI());
    Assert.assertNotNull(st.getFirstChild());
    Assert.assertTrue(st.getFirstChild() instanceof Text);
    final Text text = (Text) st.getFirstChild();
    Assert.assertEquals(name, text.getData());
  }

}
