/**
 * Copyright (C) 2010  Bull S. A. S.
 * Bull, Rue Jean Jaures, B.P.68, 78340, Les Clayes-sous-Bois
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation
 * version 2.1 of the License.
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA  02110-1301, USA.
 **/
package org.ow2.orchestra.cxf.test.camel.receive;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.util.HashMap;
import java.util.Map;

import javax.xml.namespace.QName;
import javax.xml.soap.MessageFactory;
import javax.xml.soap.SOAPBodyElement;
import javax.xml.soap.SOAPMessage;

import junit.framework.Assert;

import org.ow2.orchestra.facade.exception.OrchestraRuntimeException;
import org.ow2.orchestra.test.remote.RemoteTestCase;
import org.ow2.orchestra.test.remote.helloworld.HelloworldTest;
import org.ow2.orchestra.util.Misc;
import org.ow2.orchestra.util.SOAPUtil;
import org.ow2.orchestra.util.XmlUtil;
import org.w3c.dom.Element;
import org.w3c.dom.Text;

/**
 * @author Guillaume Porcher
 * Based on {@link HelloworldTest} by Charles Souillard
 */
public class CamelReceiveTest extends RemoteTestCase {
  private final String processName = "helloworld";
  private final String processNamespace = "http://example.com/helloworld";
  /**
   * test.
   */
  public void testCamelFileHelloworld() {
    try {
      final QName processQName = new QName(this.processNamespace, this.processName);

      // Create tmpDir
      final File tmpDir = Misc.createTempFile("orchCamel", null, null);
      tmpDir.delete();
      tmpDir.mkdir();

      // Create in dir to send requests
      final File inDir = new File(tmpDir, "in");
      inDir.mkdir();
      // Create outdir to read results
      final File outDir = new File(tmpDir, "out");
      outDir.mkdir();

      // Set in and out directories in the camel context definition

      byte[] camelContextXml = Misc.getAllContentFrom(this.getClass().getResource("camel-context.xml"));
      camelContextXml = new String(camelContextXml).replace("${fileUrl}",
          "file://" + inDir.getAbsolutePath() + "?noop=true").getBytes();
      camelContextXml = new String(camelContextXml).replace("${outFileUrl}", "file://" + outDir.getAbsolutePath()).getBytes();

      // Deploy the process (with the camel-context.xml file)
      final Map<String, byte[]> resources = new HashMap<String, byte[]>();
      resources.put("camel-context.xml", camelContextXml);
      this.deploy(this.getClass().getResource(this.processName + ".bpel"), null, resources);

      // Make a first call to the process
      this.callService(tmpDir, inDir, outDir, "Dupont");
      // Make a second call to the process
      this.callService(tmpDir, inDir, outDir, "Toto");

      // delete temp directories
      outDir.delete();
      inDir.delete();
      tmpDir.delete();

      // undeploy process
      this.undeploy(processQName);
    } catch (final Exception e) {
      throw new OrchestraRuntimeException(e);
    }
  }

  private void callService(final File tmpDir, final File inDir, final File outDir, final String name) throws Exception {
    // Prepare message
    final String part = "<submit xmlns='http://example.com/helloworld'><st>" + name + "</st></submit>";
    final SOAPMessage request = SOAPUtil.buildDocumentSOAPMessage(
        "http://orchestra.ow2.org/helloworld/submit", part);

    // Write message to file in send directory
    final File inFile = new File(inDir, "request-" + name);
    final FileOutputStream outputStream = new FileOutputStream(inFile);
    request.writeTo(outputStream);
    outputStream.close();

    // Wait for message to be processed by orchestra-camel
    while (outDir.listFiles().length == 0) {
      Thread.sleep(200);
    }
    // Wait for reply to be written
    Thread.sleep(1000);

    // Check result directory
    final File[] outFiles = outDir.listFiles();
    Assert.assertEquals(1, outFiles.length);


    // Read reply message
    final FileInputStream fileInputStream = new FileInputStream(outFiles[0]);
    final SOAPMessage response = MessageFactory.newInstance().createMessage(null, fileInputStream);
    fileInputStream.close();

    // check response
    if (response != null) {

      final SOAPBodyElement result = this.getSOAPBodyElement(response);
      this.checkReturn(result, name);
    } else {
      Assert.assertTrue("No return available", false);
    }

    // delete result and inFile
    outFiles[0].delete();
    inFile.delete();
  }

  private void checkReturn(final SOAPBodyElement result, final String name) {
    Assert.assertNotNull(result);
    Assert.assertEquals(XmlUtil.toString(result), "submit", result.getElementName().getLocalName());
    Assert.assertEquals(XmlUtil.toString(result), this.processNamespace, result.getElementName().getURI());

    Assert.assertNotNull(XmlUtil.toString(result), result.getFirstChild());
    Assert.assertNotNull(XmlUtil.element(result));
    final Element st = XmlUtil.element(result);
    Assert.assertEquals("st", st.getLocalName());
    Assert.assertEquals(XmlUtil.toString(st), this.processNamespace, st.getNamespaceURI());
    Assert.assertNotNull(st.getFirstChild());
    Assert.assertTrue(st.getFirstChild() instanceof Text);
    final Text text = (Text) st.getFirstChild();
    Assert.assertEquals("hello " + name, text.getData());
  }

}
