/**
 * Copyright (C) 2009  Bull S. A. S.
 * Bull, Rue Jean Jaures, B.P.68, 78340, Les Clayes-sous-Bois
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation
 * version 2.1 of the License.
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA  02110-1301, USA.
 **/
package org.ow2.orchestra.cxf.test.rm;

import junit.framework.Assert;
import org.ow2.orchestra.facade.exception.OrchestraRuntimeException;
import org.ow2.orchestra.test.remote.RemoteTestCase;
import org.ow2.orchestra.util.Misc;
import org.ow2.orchestra.util.SOAPUtil;
import org.ow2.orchestra.util.XmlConstants;
import org.ow2.orchestra.util.XmlUtil;
import org.w3c.dom.Element;
import org.w3c.dom.Text;

import javax.xml.namespace.QName;
import javax.xml.soap.SOAPBodyElement;
import javax.xml.soap.SOAPElement;
import javax.xml.soap.SOAPFault;
import javax.xml.soap.SOAPMessage;

/**
 * @author Guillaume Porcher
 */
public class WSRMTest extends RemoteTestCase {
  /**
   *
   */
  private static final String WS_ADD_2004_ANONYMOUS = "http://schemas.xmlsoap.org/ws/2004/08/addressing/role/anonymous";
  private final String helloProcessName = "helloworld";
  private final String helloProcessNamespace = "http://orchestra.ow2.org/cxf/test/wsrm";

  private final String invokerProcessName = "invoker";
  private final String invokerProcessNamespace = "http://orchestra.ow2.org/cxf/test/wsrm/invoker";

  private static final String WS_RM_NS_URI = "http://schemas.xmlsoap.org/ws/2005/02/rm";
  /**
   * test helloWorld: process configured to require WS-RM.
   * Check receive activities work with WS-RM
   */
  public void testHelloworldWithRM() {
    try {
      final QName processQName = new QName(this.helloProcessNamespace, this.helloProcessName);

      this.deploy(this.helloProcessName);


      /*
       * First call: no ws-rm sequence -> fail
       */
      String part = "<submit xmlns='http://orchestra.ow2.org/cxf/test/wsrm'><st>Dupont</st></submit>";
      SOAPMessage request = SOAPUtil.buildDocumentSOAPMessage(
          "http://orchestra.ow2.org/cxf/test/wsrm/helloworldPT/submitRequest", part);

      final String endpoint = this.getDefaultEndpoint("helloworldPort");

      SOAPMessage response = SOAPUtil.call(request, endpoint);
      if (response != null) {
        final SOAPBodyElement result = this.getSOAPBodyElement(response);
        Assert.assertTrue(result instanceof SOAPFault);
      } else {
        Assert.assertTrue("No return available", false);
      }

      /*
       * Second call: ws-rm sequence -> success
       */
      part = "<wsrm:CreateSequence xmlns:wsrm='" + WSRMTest.WS_RM_NS_URI + "' xmlns:wsa='" + XmlConstants.ADDRESSING_2004_REF_NAMESPACE + "'>"
           + "<wsrm:AcksTo><wsa:Address>" + WSRMTest.WS_ADD_2004_ANONYMOUS +"</wsa:Address></wsrm:AcksTo>"
           + "<wsrm:Offer><wsrm:Identifier>"+Misc.getRandomString(22)+"</wsrm:Identifier></wsrm:Offer>"
           + "</wsrm:CreateSequence>";
      request = SOAPUtil.buildDocumentSOAPMessage(
          "http://schemas.xmlsoap.org/ws/2005/02/rm/CreateSequence", part);
      request.getSOAPHeader().addNamespaceDeclaration("wsa", XmlConstants.ADDRESSING_2004_REF_NAMESPACE);
      request.getSOAPHeader().addChildElement("To", "wsa").addTextNode(endpoint);
      request.getSOAPHeader().addChildElement("ReplyTo", "wsa").addChildElement("Address", "wsa").addTextNode(WSRMTest.WS_ADD_2004_ANONYMOUS);
      request.getSOAPHeader().addChildElement("From", "wsa").addChildElement("Address", "wsa").addTextNode(WSRMTest.WS_ADD_2004_ANONYMOUS);
      SOAPElement wsaAction = request.getSOAPHeader().addChildElement("Action", "wsa");
      wsaAction.addTextNode("http://schemas.xmlsoap.org/ws/2005/02/rm/CreateSequence");
      SOAPElement wsaMessageId = request.getSOAPHeader().addChildElement("MessageID", "wsa");
      wsaMessageId.addTextNode(Misc.getRandomString(42));
      request.saveChanges();
      response = SOAPUtil.call(request, endpoint);

      String sequenceIdentifier = null;
      if (response != null) {
        final SOAPBodyElement result = this.getSOAPBodyElement(response);
        Assert.assertNotNull(result);
        Assert.assertEquals(XmlUtil.toString(result), "CreateSequenceResponse", result.getElementName().getLocalName());
        Assert.assertEquals(XmlUtil.toString(result), WSRMTest.WS_RM_NS_URI, result.getElementName().getURI());
        sequenceIdentifier = XmlUtil.element(result, WSRMTest.WS_RM_NS_URI, "Identifier").getTextContent();
      } else {
        Assert.assertTrue("No return available", false);
      }

      part = "<submit xmlns='http://orchestra.ow2.org/cxf/test/wsrm'><st>Toto</st></submit>";
      request = SOAPUtil.buildDocumentSOAPMessage(
          "http://orchestra.ow2.org/cxf/test/wsrm/helloworldPT/submitRequest", part);
      request.getSOAPHeader().addNamespaceDeclaration("wsa", XmlConstants.ADDRESSING_2004_REF_NAMESPACE);
      request.getSOAPHeader().addNamespaceDeclaration("wsrm", WSRMTest.WS_RM_NS_URI);
      request.getSOAPHeader().addChildElement("To", "wsa").addTextNode(endpoint);
      request.getSOAPHeader().addChildElement("ReplyTo", "wsa").addChildElement("Address", "wsa").addTextNode(WSRMTest.WS_ADD_2004_ANONYMOUS);
      request.getSOAPHeader().addChildElement("From", "wsa").addChildElement("Address", "wsa").addTextNode(WSRMTest.WS_ADD_2004_ANONYMOUS);
      wsaAction = request.getSOAPHeader().addChildElement("Action", "wsa");
      wsaAction.addTextNode("http://orchestra.ow2.org/cxf/test/wsrm/helloworldPT/submitRequest");
      wsaMessageId = request.getSOAPHeader().addChildElement("MessageID", "wsa");
      wsaMessageId.addTextNode(Misc.getRandomString(42));
      SOAPElement rmSequence = request.getSOAPHeader().addChildElement("Sequence", "wsrm");
      rmSequence.addChildElement("Identifier", "wsrm").addTextNode(sequenceIdentifier);
      rmSequence.addChildElement("MessageNumber", "wsrm").addTextNode("1");
      request.saveChanges();
      response = SOAPUtil.call(request, endpoint);
      if (response != null) {
        final SOAPBodyElement result = (SOAPBodyElement) response.getSOAPBody()
            .getChildElements().next();
        this.checkReturn(result, "Toto");
      } else {
        Assert.assertTrue("No return available", false);
      }

      part = "<submit xmlns='http://orchestra.ow2.org/cxf/test/wsrm'><st>Titi</st></submit>";
      request = SOAPUtil.buildDocumentSOAPMessage(
          "http://orchestra.ow2.org/cxf/test/wsrm/helloworldPT/submitRequest", part);
      request.getSOAPHeader().addNamespaceDeclaration("wsa", XmlConstants.ADDRESSING_2004_REF_NAMESPACE);
      request.getSOAPHeader().addNamespaceDeclaration("wsrm", WSRMTest.WS_RM_NS_URI);
      request.getSOAPHeader().addChildElement("To", "wsa").addTextNode(endpoint);
      request.getSOAPHeader().addChildElement("ReplyTo", "wsa").addChildElement("Address", "wsa").addTextNode(WSRMTest.WS_ADD_2004_ANONYMOUS);
      request.getSOAPHeader().addChildElement("From", "wsa").addChildElement("Address", "wsa").addTextNode(WSRMTest.WS_ADD_2004_ANONYMOUS);
      wsaAction = request.getSOAPHeader().addChildElement("Action", "wsa");
      wsaAction.addTextNode("http://orchestra.ow2.org/cxf/test/wsrm/helloworldPT/submitRequest");
      wsaMessageId = request.getSOAPHeader().addChildElement("MessageID", "wsa");
      wsaMessageId.addTextNode(Misc.getRandomString(42));
      rmSequence = request.getSOAPHeader().addChildElement("Sequence", "wsrm");
      rmSequence.addChildElement("Identifier", "wsrm").addTextNode(sequenceIdentifier);
      rmSequence.addChildElement("MessageNumber", "wsrm").addTextNode("2");
      rmSequence.addChildElement("LastMessage", "wsrm");
      request.saveChanges();
      response = SOAPUtil.call(request, endpoint);
      if (response != null) {
        final SOAPBodyElement result = (SOAPBodyElement) response.getSOAPBody()
            .getChildElements().next();
        this.checkReturn(result, "Titi");
      } else {
        Assert.assertTrue("No return available", false);
      }


      part = "<wsrm:TerminateSequence xmlns:wsrm='" + WSRMTest.WS_RM_NS_URI + "' >"
        + "<wsrm:Identifier>" + sequenceIdentifier + "</wsrm:Identifier>"
        + "</wsrm:TerminateSequence>";
      request = SOAPUtil.buildDocumentSOAPMessage(
           "http://schemas.xmlsoap.org/ws/2005/02/rm/TerminateSequence", part);
      request.getSOAPHeader().addNamespaceDeclaration("wsa", XmlConstants.ADDRESSING_2004_REF_NAMESPACE);
      request.getSOAPHeader().addChildElement("To", "wsa").addTextNode(endpoint);
      request.getSOAPHeader().addChildElement("ReplyTo", "wsa").addChildElement("Address", "wsa").addTextNode(WSRMTest.WS_ADD_2004_ANONYMOUS);
      request.getSOAPHeader().addChildElement("From", "wsa").addChildElement("Address", "wsa").addTextNode(WSRMTest.WS_ADD_2004_ANONYMOUS);
      wsaAction = request.getSOAPHeader().addChildElement("Action", "wsa");
      wsaAction.addTextNode("http://schemas.xmlsoap.org/ws/2005/02/rm/TerminateSequence");
      wsaMessageId = request.getSOAPHeader().addChildElement("MessageID", "wsa");
      wsaMessageId.addTextNode(Misc.getRandomString(42));
      request.saveChanges();
      response = SOAPUtil.call(request, endpoint);

      this.undeploy(processQName);
    } catch (final Exception e) {
      throw new OrchestraRuntimeException(e);
    }
  }

  /**
   * test invoker: process configured to invoke helloworld with WS-RM.
   * Check invoke activities work with WS-RM.
   */
  public void testInvokerWithRM() {
    try {
      final QName helloProcessQName = new QName(this.helloProcessNamespace, this.helloProcessName);
      final QName invokerProcessQName = new QName(this.invokerProcessNamespace, this.invokerProcessName);

      this.deploy(this.helloProcessName);
      this.deploy(this.invokerProcessName);


      /*
       * First call: no addressing action in header -> ok
       */
      final String part = "<submit xmlns='http://orchestra.ow2.org/cxf/test/wsrm'><st>Dupont</st></submit>";
      final SOAPMessage request = SOAPUtil.buildDocumentSOAPMessage(
          "http://orchestra.ow2.org/cxf/test/wsrm/invoker/invokerPT/submitRequest", part);

      final String endpoint = this.getDefaultEndpoint("invokerPort");

      final SOAPMessage response = SOAPUtil.call(request, endpoint);
      if (response != null) {
        final SOAPBodyElement result = this.getSOAPBodyElement(response);
        this.checkReturn(result, "Dupont");
      } else {
        Assert.assertTrue("No return available", false);
      }

      this.undeploy(invokerProcessQName);
      this.undeploy(helloProcessQName);
    } catch (final Exception e) {
      throw new OrchestraRuntimeException(e);
    }
  }

  private void checkReturn(final SOAPBodyElement result, final String name) {
    Assert.assertNotNull(result);
    Assert.assertEquals(XmlUtil.toString(result), "submit", result.getElementName().getLocalName());
    Assert.assertEquals(XmlUtil.toString(result), this.helloProcessNamespace, result.getElementName().getURI());

    Assert.assertNotNull(XmlUtil.toString(result), result.getFirstChild());
    Assert.assertNotNull(XmlUtil.element(result));
    final Element st = XmlUtil.element(result);
    Assert.assertEquals("st", st.getLocalName());
    Assert.assertEquals(XmlUtil.toString(st), this.helloProcessNamespace, st.getNamespaceURI());
    Assert.assertNotNull(st.getFirstChild());
    Assert.assertTrue(st.getFirstChild() instanceof Text);
    final Text text = (Text) st.getFirstChild();
    Assert.assertEquals(name, text.getData());
  }

}
